function M = tutorial_biquad_module(NAME)
% M = tutorial_biquad_module(NAME)
% Example showing how to implement a multichannel Biquad filter in Audio Weaver.
% Arguments:
%    NAME - name of the module.

% Copyright 2020.  DSP Concepts, Inc.  All Rights Reserved.

% ----------------------------------------------------------------------
% Create the module object
% ----------------------------------------------------------------------

M = awe_module('TutorialBiquad', 'Multichannel Biquad filter');
if (nargin == 0)
    return;
end
M.name = NAME;

% ----------------------------------------------------------------------
% Assign module functions.
% Process function is optional and used for regression tests.
% ----------------------------------------------------------------------

M.processFunc = @tutorial_biquad_process;
M.testHarnessFunc = @test_tutorial_biquad;
M.preBuildFunc = @tutorial_biquad_prebuild;

% ----------------------------------------------------------------------
% Add input and output pins
% ----------------------------------------------------------------------

PT = new_pin_type([], [], []);
add_pin(M, 'input', 'in', 'Input signal', PT);
add_pin(M, 'output', 'out', 'Output signal', PT);

% ----------------------------------------------------------------------
% Add module variables
% ----------------------------------------------------------------------

add_variable(M, 'b0', 'float', 1, 'parameter', 'First numerator coefficient');
add_variable(M, 'b1', 'float', 0, 'parameter', 'Second numerator coefficient');
add_variable(M, 'b2', 'float', 0, 'parameter', 'Third numerator coefficient');
add_variable(M, 'a1', 'float', 0, 'parameter', 'Second denominator coefficient');
add_variable(M, 'a2', 'float', 0, 'parameter', 'Third denominator coefficient');

% Initialize the state array for a single channel.  This will be updated
% in the module's prebuild function below
add_array(M, 'state', 'float', [0;0], 'state', 'State variables. 2 per channel.');
M.state.arrayHeap = 'AWE_HEAP_FAST2SLOW';
M.state.arraySizeConstructor = 'ClassWire_GetChannelCount(pWires[0]) * 2 * sizeof(FLOAT32)';
M.state.isHidden = 1;

% ----------------------------------------------------------------------
% Code generation details
% ----------------------------------------------------------------------

awe_addcodemarker(M, 'processFunction', 'Insert:InnerTutorialBiquad_Process.c');

% This means that the processing can be done in-place
M.wireAllocation = 'across';

% ----------------------------------------------------------------------
% Module documentation
% ----------------------------------------------------------------------

awe_addcodemarker(M, 'discussion', {'Standard 5 coefficient Biquad filter that operates on multichannel data. ', ...
'The module uses a Transposed Direct Form 2 implementation using the difference equation: ', ...
'', ...
'  wN = a1*wNm1 + a2*wNm2 + x[n]', ...
'y[n] = b0*wN + b1*wNm1 + b2*wNm2', ...
'wNm2 = wNm1', ...
'wNm1 = wN', ...
'', ...
'This module is an example to be used with the documentation.'});

% ----------------------------------------------------------------------
% Add the inspector information
% ----------------------------------------------------------------------

add_control(M, 'b0');
add_control(M, 'b1');
add_control(M, 'b2');
add_control(M, 'a1');
add_control(M, 'a2');

% ----------------------------------------------------------------------
% Module browser information
% ----------------------------------------------------------------------

M.moduleBrowser.path = 'Tutorial';
M.moduleBrowser.image = '../images/Tutorial.bmp';
M.moduleBrowser.searchTags = 'filter';
M.shapeInfo.basicShape = 'rectangle';
M.shapeInfo.legend = 'Biquad';

% ----------------------------------------------------------------------
% Pin function.  This changes the size of the state variables based
% on the number of channels in the input.  We also make the output
% pin the same type as the input pin.
% ----------------------------------------------------------------------

function M = tutorial_biquad_prebuild(M)

% Set the size of the state variables based on the number of channels.
M.state.size = [2 M.inputPin{1}.type.numChannels];
M.state = zeros(2, M.inputPin{1}.type.numChannels);

% Propogate the input pin type to the output pin
% This copies numChannels, blockSize, and sampleRate
M.outputPin{1}.type = M.inputPin{1}.type;

return;