/****************************************************************************
*
*               Audio Framework
*               ---------------
*
****************************************************************************
*     ModTutorialPeakingFilter.c
****************************************************************************
*
*     Description:  Multichannel Biquad filter
*
*     Copyright:    2019 DSP Concepts, Inc. All rights reserved.
*                   3235 Kifer Road
*                   Santa Clara, CA 95054
*
***************************************************************************/

/**
 * @addtogroup Modules
 * @{
 */

/**
 * @file
 * @brief Multichannel Biquad filter
 */

#define NOREDEF


#include "Framework.h"
#include "Errors.h"
#include "ModTutorialPeakingFilter.h"


#ifdef __cplusplus
extern "C" {
#endif

/* ----------------------------------------------------------------------
** THIS IS AN AUTO-GENERATED FILE. Please make all changes in the inner file. 
** Changes made to this file will be overwritten during module generation.
** Audio module class object.  This describes the audio module to the
** framework.  It contains pointers to functions and number of
** variables.
** ------------------------------------------------------------------- */

CREATE_MODULE_CLASS(Class_awe_modTutorialPeakingFilter, (5 + 3))

AWE_MOD_SLOW_ANY_CONST
const Class_awe_modTutorialPeakingFilter awe_modTutorialPeakingFilterClass =
{
    {
        { awe_modTutorialPeakingFilterConstructor, CLASSID_TUTORIALPEAKINGFILTER, },
        awe_modTutorialPeakingFilterProcess,                // Processing function
        IOMatchUpModule_Bypass,                 // Bypass function
        awe_modTutorialPeakingFilterSet,                    // Set function
        0,                                    // Get function
        0,				                      // Unused field
        ClassModule_PackArgCounts(5, 3),    // (Public words, private words)
        {0x000000FF, 0x00000000}, // Specifies which variables are floating-point
    },
#ifdef BUILD64
    {
        offsetof(awe_modTutorialPeakingFilterInstance, freq),
        offsetof(awe_modTutorialPeakingFilterInstance, gain),
        offsetof(awe_modTutorialPeakingFilterInstance, Q),
        offsetof(awe_modTutorialPeakingFilterInstance, smoothingTime),
        offsetof(awe_modTutorialPeakingFilterInstance, smoothingCoeff),
        offsetof(awe_modTutorialPeakingFilterInstance, coeffs),
        offsetof(awe_modTutorialPeakingFilterInstance, currentCoeffs),
        offsetof(awe_modTutorialPeakingFilterInstance, state),
    }
#endif
};


/* ----------------------------------------------------------------------
** Memory allocation function.  This is required because the module
** requires additional memory outside of its instance structure.
** ------------------------------------------------------------------- */

AWE_MOD_SLOW_CODE
ModInstanceDescriptor *awe_modTutorialPeakingFilterConstructor(INT32 * FW_RESTRICT retVal, UINT32 nIO, WireInstance ** FW_RESTRICT pWires, size_t argCount, const Sample * FW_RESTRICT args)
{   
	awe_modTutorialPeakingFilterInstance *S = (awe_modTutorialPeakingFilterInstance *) BaseClassModule_Constructor((ModClassModule *) &awe_modTutorialPeakingFilterClass, retVal, nIO, pWires, argCount, args);
	
	// Check if BaseClassModule_Constructor() finished properly.  If not,
	// the error code is in *retVal
	if (S == NULL)
	{
	    return 0;
	}
	
    if ((S->coeffs = (FLOAT32 *) awe_fwMalloc(sizeof(FLOAT32) * 5, AWE_HEAP_FAST2SLOW, retVal)) == 0)            
    {            
        // Error code is in *retVal            
        return 0;            
    }            
                
    if ((S->currentCoeffs = (FLOAT32 *) awe_fwMalloc(sizeof(FLOAT32) * 5, AWE_HEAP_FAST2SLOW, retVal)) == 0)            
    {            
        // Error code is in *retVal            
        return 0;            
    }            
                
    if ((S->state = (FLOAT32 *) awe_fwMalloc(ClassWire_GetChannelCount(pWires[0]) * 2 * sizeof(FLOAT32), AWE_HEAP_FAST2SLOW, retVal)) == 0)            
    {            
        // Error code is in *retVal            
        return 0;            
    }            
                
                
    return ((ModInstanceDescriptor *) S);            

    
}

/* ----------------------------------------------------------------------
** Real-time Processing function.
** ------------------------------------------------------------------- */

AWE_MOD_FAST_CODE
void awe_modTutorialPeakingFilterProcess(void *pInstance)
{
    awe_modTutorialPeakingFilterInstance *S = (awe_modTutorialPeakingFilterInstance *)pInstance;
    WireInstance **pWires = ClassModule_GetWires(S);
    INT32 blockSize = ClassWire_GetBlockSize(pWires[0]);
    INT32 numChannels = ClassWire_GetChannelCount(pWires[0]);
    INT32 n;
    INT32 chan;
    FLOAT32 wN, wNm1, wNm2;
    FLOAT32 b0, b1, b2, a1, a2;
    FLOAT32 *src, *dst;
    
    // Update the filter coefficients once at the start of processing
    
    {
        FLOAT32 smoothingCoeff = S->smoothingCoeff;
        INT32 i;
    
        for (i = 0; i < 5; i++)
        {
            S->currentCoeffs[i] = S->currentCoeffs[i] * (1.0f - smoothingCoeff) + S->coeffs[i] * smoothingCoeff;
        }
    }
    
    // Pull out module variables into local registers
    
    b0 = S->currentCoeffs[0];
    b1 = S->currentCoeffs[1];
    b2 = S->currentCoeffs[2];
    a1 = -S->currentCoeffs[3];
    a2 = -S->currentCoeffs[4];
    
    for(chan = 0; chan < numChannels; chan++)
    {
        src = ((FLOAT32 *) pWires[0]->buffer) + chan;
        dst = ((FLOAT32 *) pWires[1]->buffer) + chan;
    
        wNm1 = S->state[2*chan + 0];
        wNm2 = S->state[2*chan + 1];
    
        for(n = 0; n < blockSize; n++)
        {
            wN = (a1*wNm1) + (a2*wNm2) + (*src);
            *dst =  (b0*wN) + (b1*wNm1) + (b2*wNm2);
            wNm2 = wNm1;
            wNm1 = wN;
    
            src += numChannels;
            dst += numChannels;
        }
    
        S->state[2*chan + 0] = wNm1;
        S->state[2*chan + 1] = wNm2;
    }
    
}

/* ----------------------------------------------------------------------
** Set function which updates derived parameters based on the
** module's interface variables.
** ------------------------------------------------------------------- */

AWE_MOD_SLOW_CODE
UINT32 awe_modTutorialPeakingFilterSet(void *pInstance, UINT32 mask)
{
    awe_modTutorialPeakingFilterInstance *S = (awe_modTutorialPeakingFilterInstance *) pInstance;
    WireInstance **pWires = ClassModule_GetWires(S);
    FLOAT32 sampleRate = ClassWire_GetSampleRate(pWires[0]);
    INT32 blockSize = ClassWire_GetBlockSize(pWires[0]);
    FLOAT32 A, alpha, w0;
    FLOAT32 b0, b1, b2, a0, a1, a2;
        
    // Compute the filter coefficients based on {freq, gain, Q}
    // Taken from Audio-EQ-Cookbook.txt, by Robert Bristow-Johnson
        
    if (mask & (MASK_TutorialPeakingFilter_freq | MASK_TutorialPeakingFilter_gain | MASK_TutorialPeakingFilter_Q))
    {
        w0 = 6.283185307179586f * S->freq / sampleRate;
        A = undb20f(S->gain * 0.5f);
        
        if (S->Q >= 0)
        {
            alpha = sinf(w0) / (2.0f * S->Q);
        }
        else
        {
            alpha = sinf(w0) * sinhf((logf(2.0f)/2.0f)*(-S->Q)*w0/sinf(w0));
        }
        
        b0 =   1.0f + alpha*A;
        b1 =  -2.0f * cosf(w0);
        b2 =   1.0f - alpha*A;
        a0 =   1.0f + alpha/A;
        a1 =  -2.0f * cosf(w0);
        a2 =   1.0f - alpha/A;
         
        // Normalize so that the a0 coefficient equals 1.0
        S->coeffs[0] = b0 / a0;
        S->coeffs[1] = b1 / a0;
        S->coeffs[2] = b2 / a0;
        S->coeffs[3] = a1 / a0;
        S->coeffs[4] = a2 / a0;
    }
        
    // Compute the smoothing coefficient based on the smoothingTime
    if (mask & MASK_TutorialPeakingFilter_smoothingTime)
    {
        S->smoothingCoeff = 1.0f - expf(-1.0f/((sampleRate / blockSize) * 0.001f * S->smoothingTime));    
    }
        
    return 0;
}



#ifdef __cplusplus
}
#endif

/**
 * @}
 *
 * End of file.
 */
