function M = tutorial_biquad_smoothed_module(NAME)
% M = tutorial_biquad_smoothed_module(NAME)
% Example showing how to implement a multichannel Biquad filter with
% coefficient smoothing.  Arguments:
%    NAME - name of the module.

% Copyright 2020.  DSP Concepts, Inc.  All Rights Reserved.

% ----------------------------------------------------------------------
% Create the module object
% ----------------------------------------------------------------------

M = awe_module('TutorialBiquadSmoothed', 'Multichannel Biquad filter');
if (nargin == 0)
    return;
end
M.name = NAME;

% ----------------------------------------------------------------------
% Assign module functions.
% Process function is optional and used for regression tests.
% ----------------------------------------------------------------------

M.processFunc = @tutorial_biquad_smoothed_process;
M.testHarnessFunc = @test_tutorial_biquad_smoothed;
M.preBuildFunc = @tutorial_biquad_smoothed_prebuild;
M.setFunc = @tutorial_biquad_smoothed_set;

% ----------------------------------------------------------------------
% Add input and output pins
% ----------------------------------------------------------------------

PT = new_pin_type([], [], []);
add_pin(M, 'input', 'in', 'Input signal', PT);
add_pin(M, 'output', 'out', 'Output signal', PT);

% ----------------------------------------------------------------------
% Add module variables
% ----------------------------------------------------------------------

add_variable(M, 'smoothingTime', 'float', 10, 'parameter', 'Time constant of the smoothing process.');
M.smoothingTime.units = 'msec';
M.smoothingTime.range = [0 1000];

add_variable(M, 'smoothingCoeff', 'float', 1, 'derived', 'Smoothing coefficient. This is computed based on the smoothingTime, sample rate, and block size of the module.');
M.smoothingCoeff.isHidden = 1;

add_array(M, 'coeffs', 'float', [1; 0; 0; 0; 0], 'parameter', 'Filter coefficients [b0; b1; b2; a1; a2]');
add_array(M, 'currentCoeffs', 'float', [1; 0; 0; 0; 0], 'state', 'Smoothed filter coefficients [b0; b1; b2; a1; a2]');
M.currentCoeffs.isHidden = 1;

% Initialize the state array for a single channel.  This will be updated
% in the module's prebuild function below
add_array(M, 'state', 'float', [0;0], 'state', 'State variables. 2 per channel.');
M.state.arrayHeap = 'AWE_HEAP_FAST2SLOW';
M.state.arraySizeConstructor = 'ClassWire_GetChannelCount(pWires[0]) * 2 * sizeof(FLOAT32)';
M.state.isHidden = 1;

% Call the set function.  This provides a reasonable starting value for the
% smoothing Coeff
M = update(M);

% ----------------------------------------------------------------------
% Code generation details
% ----------------------------------------------------------------------

awe_addcodemarker(M, 'processFunction', 'Insert:InnerTutorialBiquadSmoothed_Process.c');
awe_addcodemarker(M, 'setFunction', 'Insert:\InnerTutorialBiquadSmoothed_Set.c');

% This means that the processing can be done in-place
M.wireAllocation = 'across';

% ----------------------------------------------------------------------
% Module documentation
% ----------------------------------------------------------------------

awe_addcodemarker(M, 'discussion', {'Standard 5 coefficient Biquad filter that operates on multichannel data. ', ...
'The module uses a Transposed Direct Form 2 implementation using the difference equation: ', ...
'', ...
'  wN = a1*wNm1 + a2*wNm2 + x[n]', ...
'y[n] = b0*wN + b1*wNm1 + b2*wNm2', ...
'wNm2 = wNm1', ...
'wNm1 = wN', ...
'', ...
'The module has built in smoothing on a block-by-block basis. ', ...
'The user sets the .coeffs and the processing function updates the .currentCoeffs at the start of each block. ', ...
'', ...
'This module is an example to be used with the documentation.'});

% ----------------------------------------------------------------------
% Add the inspector information
% ----------------------------------------------------------------------

add_control(M, 'coeffs');
add_control(M, 'smoothingTime');

% ----------------------------------------------------------------------
% Module browser information
% ----------------------------------------------------------------------

M.moduleBrowser.path = 'Tutorial';
M.moduleBrowser.image = '../images/Tutorial.bmp';
M.moduleBrowser.searchTags = 'filter';
M.shapeInfo.basicShape = 'rectangle';
M.shapeInfo.legend = 'Biquad';

% ----------------------------------------------------------------------
% Pin function.  This changes the size of the state variables based
% on the number of channels in the input.  We also make the output
% pin the same type as the input pin.
% ----------------------------------------------------------------------

function M = tutorial_biquad_smoothed_prebuild(M)

% Have the module start in a converged state
M.currentCoeffs = M.coeffs;

% Set the size of the state variables based on the number of channels.
M.state.size = [2 M.inputPin{1}.type.numChannels];
M.state = zeros(2, M.inputPin{1}.type.numChannels);

% Propogate the input pin type to the output pin
% This copies numChannels, blockSize, and sampleRate
M.outputPin{1}.type = M.inputPin{1}.type;

return;

% ----------------------------------------------------------------------
% Set function.
% ----------------------------------------------------------------------

function M = tutorial_biquad_smoothed_set(M)

% Compute the smoothing coefficient based on the smoothing time
sampleRate = M.inputPin{1}.type.sampleRate;
blockSize = M.inputPin{1}.type.blockSize;

M.smoothingCoeff = 1.0 - exp(-1.0/((sampleRate / blockSize) * 0.001 * M.smoothingTime));

return;