/*******************************************************************************
*
*				Audio Framework
*				---------------
*
********************************************************************************
*	  Framework.h
********************************************************************************
*
*	  Description:	AudioWeaver Framework main header file
*
*	  Copyright:	(c) 2018 DSP Concepts, Inc. All rights reserved.
*					3235 Kifer Road
*					Santa Clara, CA 95054
*
*******************************************************************************/
#ifndef _FRAMEWORK_H
#define _FRAMEWORK_H

#include <stdlib.h>
#include <string.h>
#include <stddef.h>

#if defined(IA610)
#include "DeltaPlatform.h"
#include "hemilite_fr32_ops.h"
#include "hemilite_fr32_math.h"
#endif

#if defined(DMX)
#include "DeltaPlatform.h"
#include "fr32_ops.h"
#include "fr32_math.h"
#endif

#if defined(HMD)
#include "DeltaPlatform.h"
#include "fr32_ops.h"
#include "fr32_math.h"
#endif

#include "TargetProcessor.h"

#ifdef	__cplusplus
extern "C" {
#endif

#if defined(_MSC_VER) && defined(USEFRAMEWORKDLL)
#	define DLLSYMBOL			__declspec(dllimport)
#	define CXX_NOWARN_DLLCLASSIF		__pragma(warning(push))\
		__pragma(warning(disable: 4251))
#	define CXX_RESTORE_DLLCLASSIF		__pragma(warning(pop))
#else
#	define DLLSYMBOL
#define CXX_NOWARN_DLLCLASSIF
#define CXX_RESTORE_DLLCLASSIF
#endif

/** Number of samples in a DMA buffer - bound wires must be an integer multiple of this. */
#ifndef DMA_NUM_SAMPLES
#define DMA_NUM_SAMPLES 32
#endif

#define MPS_25MHZ_COUNT (*((volatile unsigned long *)(0x40004014)))

/* --------- Class identifiers. */
#define CLASS_ID_BASE							0xBEEF0000U
#define CLASS_ID_END							0xBEEFFFFFU
#define CLASS_ID_MASK							0xFFFF0000U

#define CLASS_ID_INPUTPIN						(CLASS_ID_BASE + 1U)
#define CLASS_ID_OUTPUTPIN						(CLASS_ID_BASE + 2U)
#define CLASS_ID_PIN							(CLASS_ID_BASE + 3U)
#define CLASS_ID_LAYOUT							(CLASS_ID_BASE + 4U)

#define CLASS_ID_PINBASE						CLASS_ID_INPUTPIN
#define CLASS_ID_PINEND							CLASS_ID_PIN

/*	Class IDs for wires */
#define CLASS_ID_WIREBASE						(CLASS_ID_BASE + 0x80U)
#define CLASS_ID_WIREEND						(CLASS_ID_BASE + 0x7FFU)

#define CLASS_ID_WIRE							(CLASS_ID_WIREBASE)

/* Class IDs for modules */
#define CLASS_ID_MODBASE						(CLASS_ID_BASE + 0x800U)
#define CLASS_ID_MODEND							(CLASS_ID_END)

/* Instance name IDs */
#define NAME_BASE								0x7EEDB000U

#define NAME_INPUT_INTERLEAVED					(NAME_BASE + 1U)
#define NAME_OUTPUT_INTERLEAVED					(NAME_BASE + 2U)


/** Test if a given class is valid. */
#define IsClassValid(classID) \
	(((classID) >= CLASS_ID_BASE) && ((classID) <= CLASS_ID_END))

/** Test if a given class ID is that of a wire. */
#define IsClassWire(classID) \
	(((classID) >= CLASS_ID_WIREBASE) && ((classID) <= CLASS_ID_WIREEND))

/** Test if a given class ID is that of module. */
#define IsClassModule(classID) \
	(((classID) >= CLASS_ID_MODBASE) && ((classID) <= CLASS_ID_MODEND))

/** Test if a given class ID is that of a layout. */
#define IsClassLayout(classID) \
	((classID) == CLASS_ID_LAYOUT)

#if 0
	// The concept of pin class has gone away.
/** Test if a given class ID is that of PIN type. */
#define IsClassPin(classID) \
	(((classID) >= CLASS_ID_PINBASE) && ((classID) <= CLASS_ID_PINEND))
#endif

#define WORDSOF(x)					(sizeof(x) / sizeof(INT32))

#ifndef NUMOF
#define NUMOF(x)					(sizeof(x) / sizeof(*x))
#endif

/* Handle packed wire counts. */

/** Get input pin (wire) count from nIO. */
#define GetInputPinCount(x)			((x) & 0xffU)

/** Get output pin (wire) count from nIO. */
#define GetOutputPinCount(x)		(((x) >> 8) & 0xffU)

/** Get scratch pin (wire) count from nIO. */
#define GetScratchPinCount(x)		(((x) >> 16) & 0xffU)

#if 0		// Never used
/** Get feedback pin (wire) count from nIO. */
#define GetFeedbackPinCount(x)		(((x) >> 24) & 0xffU)
#endif

/** Get total pin (wire) count from nIO. */
#define GetWireCount(x)				( (((x) >> 24) & 0xffU) + (((x) >> 16) & 0xffU) + (((x) >> 8) & 0xffU) + ((x) & 0xffU))

#if 0		// Never used
#define MakeIOSize(nInputs, nOutputs, blockSize) \
	(((blockSize) << 16) | ((nOutputs) << 8) | (nInputs))
#endif

/*------------------ Heap declarations ------------------*/

/* For the PC, all heaps are the same. No point doing anything else really. */
/** Items accessed from the inner loop, such as Wire Buffers. */
#define AWE_HEAP_FAST		(UINT32)1U

/** Infrequently used items which could be placed in external memory. */
#define AWE_HEAP_SLOW		(UINT32)2U

/** Another fast heap which is distinct from AWE_HEAP_FAST. This is used to separate
state and coefficients, as in FIR filters. */
#define AWE_HEAP_FASTB		(UINT32)3U

/** Fast Heap -> FastB Heap -> Slow Heap */
#define AWE_HEAP_FAST2SLOW ((AWE_HEAP_FAST) | ((AWE_HEAP_FASTB) << (4)) | ((AWE_HEAP_SLOW) << (8)))

/** Fast Heap -> FastB Heap */
#define AWE_HEAP_FAST2FASTB ((AWE_HEAP_FAST) | ((AWE_HEAP_FASTB) << (4)))

/** Fast B Heap -> Fast Heap */
#define AWE_HEAP_FASTB2FAST ((AWE_HEAP_FASTB) | ((AWE_HEAP_FAST) << (4)))

/** This performs the allocation in the fast B heap, and if this fails, in the slow heap */
#define AWE_HEAP_FASTB2SLOW ((AWE_HEAP_FASTB) | ((AWE_HEAP_SLOW) << (4)))

/** Return the number of heaps. Constant at compile time. */
UINT32 awe_fwGetHeapCount(void);


/*------------------ Union declarations ------------------*/
typedef union _Sample
{
	INT32 iVal;
	UINT32 uiVal;
	float fVal;
}
Sample;

/** Defines for processor type. */
#define PROCESSOR_TYPE_NATIVE		1U
#define PROCESSOR_TYPE_SHARC		2U	// SHARC 214xx
#define PROCESSOR_TYPE_BLACKFIN		3U
#define PROCESSOR_TYPE_CORTEXM4		4U
#define PROCESSOR_TYPE_OMAP			5U
#define PROCESSOR_TYPE_DSK6713		6U
#define PROCESSOR_TYPE_HIFI2DSP		7U	// Generic HiFi 2
#define PROCESSOR_TYPE_CORTEXM3		8U
#define PROCESSOR_TYPE_IMX25		9U
#define PROCESSOR_TYPE_CORTEXM7		10U
#define PROCESSOR_TYPE_C674x		11U
#define PROCESSOR_TYPE_CortexA5		12U
#define PROCESSOR_TYPE_CortexA7		13U
#define PROCESSOR_TYPE_CortexA8		14U
#define PROCESSOR_TYPE_CortexA9		15U
#define PROCESSOR_TYPE_CortexA12	16U
#define PROCESSOR_TYPE_CortexA15	17U
#define PROCESSOR_TYPE_CortexA53	18U
#define PROCESSOR_TYPE_CortexA57	19U
#define PROCESSOR_TYPE_ARM9			20U
#define PROCESSOR_TYPE_ARM11		21U
#define PROCESSOR_TYPE_HEXAGON		22U // QUALCOMM HEXAGON
#define PROCESSOR_TYPE_HIFI3DSP		23U // Generic HiFi 3
#define PROCESSOR_TYPE_S1000		24U	// Intel S1000 for Sue Creek
#define PROCESSOR_TYPE_HEMILITE		25U	// Knowles processor in IA-610
#define PROCESSOR_TYPE_CORTEXA72	26U
#define PROCESSOR_TYPE_CORTEXA35	27U
#define PROCESSOR_TYPE_C66xx		28U
#define PROCESSOR_TYPE_SHARC215XX	29U	// SHARC 215xx family
#define PROCESSOR_TYPE_HIFI4DSP		30U // Generic HiFi 4
#define PROCESSOR_TYPE_DELTAMAX		31U // Knowles deltaMax processor
#define PROCESSOR_TYPE_HEMIDELTA	32U // Knowles hemiDelta processor
#define PROCESSOR_TYPE_CEVA			33U // CEVA processor
#define PROCESSOR_TYPE_FORDHIFI4	34U // Ford iMX8 HiFi4 processor
#define PROCESSOR_TYPE_CEVAX2    	35U // CEVA X2 processor
#define PROCESSOR_TYPE_UNASSIGNED   36U // Placeholder type
#define PROCESSOR_TYPE_CORTEXM33    37U 

/** Target has no file system. */
#define FILESYSTEM_TYPE_NONE			0U

/** Target has compiled in file system. */
#define FILESYSTEM_TYPE_COMPILED_IN		1U

/** Target has flash file system (default). */
#define FILESYSTEM_TYPE_FLASH			2U

/** Target has FAT file system */
#define FILESYSTEM_TYPE_FAT				3U

/** Create the target version - for V5 must have first argument of 5. */
#define MAKE_TARGET_VERSION(v1, v2, v3, v4) \
	( (((v1) & 0xffU) << 24) | (((v2) & 0xffU) << 16) | (((v3) & 0xffU) << 8) | ((v4) & 0xffU) )

/** Create a word containing packed characters. */
#define MAKE_PACKED_STRING(c1, c2, c3, c4) \
	( (((c4) & 0xffU) << 24) | (((c3) & 0xffU) << 16) | (((c2) & 0xffU) << 8) | ((c1) & 0xffU) )


/** This structure defines the target info. */
/*private*/ typedef struct _TargetInfo
{
	/** Target sample rate. */
	float m_sampleRate;					/* 0 */

	/** Target profile clock speed in Hz. */
	float m_profileClockSpeed;			/* 4 */

	/** Base block size of target (usually 32). */
	UINT32 m_base_block_size;			/* 8 */

	/** Packed field. */
	UINT32 m_packedData;				/* 12 */

	/** Target version - high byte is base framework version and must match. */
	UINT32 m_version;					/* 16 */

	/** Packed field for buffer size, s, input pin count, output pin count. */
	UINT32 m_proxy_buffer_size;			/* 20 */

	/** Target name up to 16 characters. */
	UINT32 m_packedName[2];				/* 24 */

	/** Clock speed of this core. */
	float m_coreClockSpeed;				/* 32 */

	/** ID of this core. */
	UINT32 m_coreID;					/* 36 */

	/** Feature bits. */
	UINT32 m_features;					/* 40 */
}
TargetInfo;

/** This structure defines extended target info. */
/* private */ typedef struct _ExtendedInfo
{
	/** The user version field. */
	UINT32 userVersion;

	/** Currently undefined fields that will report zeros. */
	UINT32 notSpecified[12];

} ExtendedInfo;	// Total length 13 words

/** Set the m_proxy_buffer_size target info member. */
#define PACKED_BUFFER_SIZE(bufferSize, cores, nInputPins, nOutputPins, smpCore) \
	(((bufferSize - 1U) & 0x1fffU) | (((cores - 1U) & 0x3fU) << 13) | (((nInputPins - 1U) & 0x3fU) << 20) | (((nOutputPins - 1U) & 0x3fU) << 26) | (((smpCore) & 1U) << 19))

/** Get the packet buffer size from m_proxy_buffer_size. */
#define GET_TARGET_PACKET_BUFFER_LEN(x) (((x).m_proxy_buffer_size & 0x1fffU) + 1U)

/** The the target API. */
#define GET_TARGET_API(x)				(((x).m_version >> 24) & 0xff)

/** Get the core count from m_proxy_buffer_size. */
#define GET_TARGET_CORES(x)				((((x).m_proxy_buffer_size >> 13) & 0x3fU) + 1U)

#define SMP_CORE_BIT	0x80000U

/** Test if the core is static. */
#define IS_SMP_CORE(x)					(((x).m_proxy_buffer_size & SMP_CORE_BIT) != 0U)

/** Get the number of input pins from m_proxy_buffer_size. */
#define GET_TARGET_NINPUT_PINS(x)		((((x).m_proxy_buffer_size >> 20) & 0x3fU) + 1U)

/** Get the number of output pins from m_proxy_buffer_size. */
#define GET_TARGET_NOUTPUT_PINS(x)		((((x).m_proxy_buffer_size >> 26) & 0x3fU) + 1U)

/** Get the base block size of the target. */
#define GET_TARGET_BASE_BLOCK_SIZE(x)	((x).m_base_block_size & 0xfffU)

/** Get the number of threads on the target. */
#define GET_TARGET_THREADS(x)			(((x).m_base_block_size >> 12) & 0xfU)

/** Set only on special targets */
#define FLOAT_IO_PINS		0x100000U

/** Test if the target uses float I/O pins. */
#define IS_IO_PIN_FLOAT(x)				(((x).m_base_block_size & FLOAT_IO_PINS) != 0U)

/** BSP must use this macro to set the bit in TargetInfo. */
#define SET_IS_PIN_FLOAT(x)				((x).m_base_block_size |= FLOAT_IO_PINS)

/** Get the clock domain of the core - 0 is main domain. */
#define GET_CLOCK_DOMAIN(x)				(((x).m_base_block_size >> 16) & 0xfU)

#define MAKE_BLOCK_SIZE_PACKED(block_size, nThreads) \
	((block_size) & 0xfffU) | (((nThreads) & 0xfU) << 12)

/** Create target info m_packedData. */
#define MAKE_TARGET_INFO_PACKED(sizeof_int, num_inputs, num_outputs, is_floating_point, is_flash_supported, processor_type) \
	( ((sizeof_int) & 0xfU) | (((is_flash_supported) & 1U) << 4) | (((is_floating_point) & 1U) << 5) \
	| (((num_inputs) & 0xffU) << 8) | (((num_outputs) & 0xffU) << 16) | (((processor_type) & 0x7fU) << 24) )

/** Access sizeof(int) from target info m_packedData. */
#define TARGET_INFO_SIZOF_INT(targetinfo) \
	((targetinfo).m_packedData & 0xfU)

/** Access flash supported from target info m_packedData. */
#define TARGET_INFO_IS_FLASH_SUPPORTED(targetinfo) \
	(((targetinfo).m_packedData & 0x10U) != 0)

/** Access is floating point from target info m_packedData. */
#define TARGET_INFO_IS_FLOATING_POINT(targetinfo) \
	(((targetinfo).m_packedData & 0x20U) != 0)

/** Bit if target uses paddin. */
#define TARGET_USES_PADDING		0x40U

/** Report if heap uses padding. */
#define TARGET_INFO_USES_PADDING(targetinfo) \
	(((targetinfo).m_packedData & TARGET_USES_PADDING) != 0)

/** Access num inputs from target info m_packedData. */
#define TARGET_INFO_NUM_INPUTS(targetinfo) \
	(((targetinfo).m_packedData >> 8) & 0xffU)

/** Access num outputs from target info m_packedData. */
#define TARGET_INFO_NUM_OUTPUTS(targetinfo) \
	(((targetinfo).m_packedData >> 16) & 0xffU)

/** Access processor type from target info m_packedData. */
#define TARGET_INFO_PROCESSOR_TYPE(targetinfo) \
	(((targetinfo).m_packedData >> 24) & 0x7fU)


/** Routing info. */
typedef struct _RouteInfo
{
	/** Packed route info. */
	UINT32 m_packed;
}
RouteInfo;

/** Construct the packed route info. */
#define ROUTE_PACK(sourceCore, sourcePinIndex, destinationCore, destinationPinIndex) \
	(((sourceCore) & 0xffU) << 24) | (((sourcePinIndex) & 0xffU) << 16) | (((destinationCore) & 0xffU) << 8) | ((destinationPinIndex) & 0xffU)

/** Get the route source core. */
#define ROUTE_SOURCE_CORE(x)		(((x).m_packed >> 24) & 0xffU)

/** Get the route source pin. */
#define ROUTE_SOURCE_PIN_INDEX(x)	(((x).m_packed >> 16) & 0xffU)

/** Get the route destination core. */
#define ROUTE_DEST_CORE(x)			(((x).m_packed >> 8) & 0xffU)

/** Get the route destination pin. */
#define ROUTE_DEST_PIN_INDEX(x)		((x).m_packed & 0xffU)


/** States for ModuleFileStreaming. */
enum FileIOStatus
{
	/* ModuleFileStreaming always writes this (zero) to handshake. */
	FIOS_Null,

	/* These states are in the high byte of buffer[buffer_size]. */
	FIOS_NewStream,
	FIOS_NextBlock,
	FIOS_LastBlock,

	/* These states are reported asynchronously in asyncStatus. */
	FIOS_Stopped,
	FIOS_Paused,
	FIOS_Error,

	/* Used internally by the file streaming module to keep track of the current state */
	FIOS_Playing,
	FIOS_PlayingLast
};

/** This structure defines the file system info. */
typedef struct _FileSystemInfo
{
	UINT32 m_FileSystemType;				/* 0 */
	UINT32 m_FlashDeviceDWords;				/* 4 */
	UINT32 m_FileSystemDWords;				/* 8 */
	UINT32 m_DataStructOverheadDWords;		/* 12 */
	UINT32 m_DeletedOrCorruptedDWords;		/* 16 */
	UINT32 m_DWordsInUse;					/* 20 */
	UINT32 m_DWordsAvailable;				/* 24 */
	UINT32 m_BlkSize_MaxFilename_Len;		/* 28 */
}
FileSystemInfo;		/* size=32 (8 words) */

/** The one and only file system info object. */
extern FileSystemInfo g_filesystem_info;

/* Forward reference. */
/*private*/ struct _AWEInstance;

/*------------------ Instance declarations ------------------*/

/** A class descriptor. Common to all classes. */
/*private*/ typedef struct _ClassDescriptor
{
	/** All constructors look the same. To make debugging easy, we pass a generic
	array of words to the constructor, which needs to figure out what to do with
	them.

	On return, we have a pointer to an initialized instance ready for use.
	*/
	struct _InstanceDescriptor *(*Constructor)(struct _AWEInstance *pAWE, INT32 *retVal, INT32 srate, UINT32 info1, UINT32 info2, UINT32 info3);

	/** Unique ID of the class - set at compile time. */
	UINT32 classID;
}
ClassDescriptor;

/** The table of classes known to the framework. */
extern const ClassDescriptor *g_class_descriptor_table[2];


/** An instance descriptor. Common to all instances. */
/*private*/ typedef struct _InstanceDescriptor
{
	/** Chain instances together in a linked list. */
	struct _InstanceDescriptor *pNextInstance;		/* 0 */

	/** Pointer back to the static descriptor from which this instance was
	created. */
	const ClassDescriptor *pClassDescriptor;		/* 4 */

	/** The unique instance ID of this instance. */
	UINT32 nUniqueInstanceID;						/* 8 */
}
InstanceDescriptor;		/* 12 bytes */

/** Description of bound wires. */
/*private*/ typedef struct _CoreWireDescr
{
	/** The core this wire is on. */
	UINT32 coreID;					/* 4 */

	/** The wire on the core - pointer in that core's address space. */
	struct _WireInstance *pWire;	/* 8 */
} CoreWireDescr;

/** An I/O pin instance. These objects are defined at compile
time to match the physical I/O. */
/*private*/ typedef struct _IOPinDescriptor
{
	/** The basic instance data. */
	InstanceDescriptor instanceDescriptor;	/* 12 */

	/** Pointer to the buffer allocated by the wire - copied from the wire at bind time. */
	Sample *pOrigBuffer;					/* 16 */

	/** Pointer to the double buffer allocated at bind time. */
	Sample *pSecondBuffer;					/* 20 */

	/** Pin sample rate. */
	float sampleRate;						/* 24 */

	// Number of channels		10 bits - 0x3ff
	// Max block size			17 bits - 0x7fffC00
	// isComplex				1 bit - 0x8000000
	// Sample size in bytes.	4 bits - 0xf0000000
	UINT32 wireInfo1;						/* 28 */

	// Block size				17 bits - 0x1ffff
	// Data Type				6 bits - 0x7e0000
	UINT32 wireInfo2;						/* 32 */

	// Rows						10 bits - 0x3ff
	// Columns					10 bits - 0xffc00
	// IsIPC					1 bit - 0x100000
	// IsPrivate				1 bit - 0x200000
	// Clock master				1 bit - 0x400000
	// Special input pin		1 bit - 0x800000
	UINT32 wireInfo3;						/* 36 */

	/** Flags to control buffer flipping. */
	UINT32 ctrlFlags;						/* 40 */

	/** Number of bound wires. */
	UINT32 nBoundWires;						/* 44 */

	/** Array of bound wires. */
	CoreWireDescr boundWires[16];			/* 44 + 16 * 8 = 172 */

	/** Optional named pin. */
	UINT32 m_pinName[2];					/* 180 */
}
IOPinDescriptor;

/** Create info1 word. */
#define INFO1_PROPS(channels, maxBlockSize, complex, nSizeBytes) \
	((channels & 0x3ffU) | ((maxBlockSize & 0x1ffffU) << 10) | ((complex & 0x1U) << 27) | ((nSizeBytes & 0xfU) << 28))

/** Create info2 word. */
#define INFO2_PROPS(blockSize, dataType) \
	((blockSize & 0x1ffffU) | ((dataType & 0x3fU) << 17))

/** Create info3 word. */
#define INFO3_PROPS(rows, cols) \
	((((cols) & 0x3ffU) << 10) | ((rows) & 0x3ffU))

/** The IPC bit in info3. */
#define IS_IPC_BIT			((UINT32)1U << 20)

/** The private pin bit in info3. */
#define IS_PRIVATE_BIT		((UINT32)1U << 21)

/** The clock master bit. */
#define CLOCK_MASTER_BIT	((UINT32)1U << 22)

/** Set on pins that are special inputs. */
#define SPECIAL_INPUT_BIT	((UINT32)1U << 23)

/** Test if pin is the clock master. */
#define INFO3_IS_CLOCK_MASTER(info3) \
	(CLOCK_MASTER_BIT & (info3)) != 0

/** Test if a pin's size is compatible with an I/O pin's size. */
#define IsCompatiblePinSize(wireBlock, pinBlock) \
	(wireBlock % pinBlock == 0)


/*------------------ Declarations of I/O ------------------*/

/** The input pins - default 1. */
extern DLLSYMBOL IOPinDescriptor InterleavedInputPin[];

/** The output pins - default 1. */
extern DLLSYMBOL IOPinDescriptor InterleavedOutputPin[];


/*------------------ Wire declarations ------------------*/
/*private*/ typedef struct _WireInstance
{
	/** The basic instance data. */
	InstanceDescriptor instanceDescriptor;

	/** The wire buffer. */
	Sample *buffer;

	/** Wire sample rate. */
	float sampleRate;

	// Number of channels		10 bits - 0x3ff
	// Max block size			17 bits - 0x7fffC00
	// isComplex				1 bit - 0x8000000
	// Sample size in bytes.	4 bits - 0xf0000000
	UINT32 wireInfo1;

	// Block size				17 bits - 0x1ffff
	// Data Type				6 bits - 0x7e0000
	UINT32 wireInfo2;

	// Rows						10 bits - 0x3ff
	// Columns					10 bits - 0xffc00
	UINT32 wireInfo3;

	/** Bind chain pointer default NULL. */
	struct _WireInstance *m_pNextBind;

	/** What its bound to default NULL. */
	IOPinDescriptor *m_pBoundPin;

#ifndef USE_TEST_PADDING
#if defined(BUILD64) || defined(AWE_STORAGE_ALIGN4)
	UINT32 padding2;
#endif
	UINT32 padding;
#endif
}
WireInstance;

/** Get the channels from info1. */
#define WireInfo1_GetChannels(x)		((x) & 0x3ffU)				// 10 bits 0-1023

/** Get the max block size from info1. */
#define WireInfo1_GetMaxBlockSize(x)	(((x) >> 10) & 0x1ffffU)	// 17 bits 0-131071

/** Get is complex from info1. */
#define WireInfo1_IsComplex(x)			(((x) >> 27) & 0x1U)		// 1 bit, 1 if true

/** Get sample width from info1. */
#if defined(TEAKLITE3) || defined(TEAKLITE4)
#define WireInfo1_SampleWidthBytes(x)	(((x) >> 29) & 0x7U)		// 4 bits 0-15 char is 16 bits on TEAKLITE3
#else
#define WireInfo1_SampleWidthBytes(x)	(((x) >> 28) & 0xfU)		// 4 bits 0-15
#endif

/** Get the block size for info2. */
#define WireInfo2_GetBlockSize(x)		((x) & 0x1ffffU)			// 17 bits, 0-131071

/** Get the data type from info2. */
#define WireInfo2_GetDataType(x)		(((x) >> 17) & 0x3fU)		// 6 bits, 0-63

/** Get the rows from info3. */
#define WireInfo3_GetRows(x)			((x) & 0x3ffU)				// 10 bits, 0-1023

/** Get the columns from info3. */
#define WireInfo3_GetCols(x)			(((x) >> 10) & 0x3ffU)		// 10 bits, 0-1023

/** Get the IPC status from info3. */
#define WireInfo3_IsIPC(x)				(((x) >> 20) & 0x1U)		// 1 bit

/** Get the block size of a wire. */
#define ClassWire_GetBlockSize(W)		(WireInfo2_GetBlockSize((W)->wireInfo2))

/** Get the channel count of a wire. */
#define ClassWire_GetChannelCount(W)	(WireInfo1_GetChannels((W)->wireInfo1))

/** Get the number of samples in a wire. */
#define ClassWire_GetNumSamples(W)		(ClassWire_GetBlockSize(W) * ClassWire_GetChannelCount(W))

/** Get the sample rate of a wire. */
#if defined(USE_AFLOAT)
#define ClassWire_GetSampleRate(W)		(float_to_afloat((W)->sampleRate) ) // FLOAT32
#else
#define ClassWire_GetSampleRate(W)		((W)->sampleRate)	// float
#endif

/** Get if the wire is complex. */
#define ClassWire_GetComplex(W)			(WireInfo1_IsComplex((W)->wireInfo1))

/** Get the wire sample width in bytes. */
#define ClassWire_GetNSampleSize(W)		(WireInfo1_SampleWidthBytes((W)->wireInfo1))

/** Get the wire number of rows. */
#define ClassWire_GetNumMatRows(W)		(WireInfo3_GetRows((W)->wireInfo3))

/** Get the wire number of columns. */
#define ClassWire_GetNumMatCols(W)		(WireInfo3_GetCols((W)->wireInfo3))


/*------------------ Module declarations ------------------*/

/*private*/ typedef struct _ModClassDescriptor
{
	/** All constructors look the same. To make debugging easy, we pass a generic
	array of words to the constructor, which needs to figure out what to do with
	them.

	On return, we have a pointer to an initialized instance ready for use.
	*/
	struct _ModInstanceDescriptor *(*Constructor)(
			INT32 * FW_RESTRICT retVal,
			UINT32 nIO,
			WireInstance **pWires,
			size_t argCount,
			const Sample * FW_RESTRICT args);		/* 0 */

	/** Unique ID of the class - set at compile time. */
	UINT32 classID;									/* 4 */
}
ModClassDescriptor;		/* 8 bytes */


/** Class descriptor for audio modules. */
/*private*/ typedef struct _ModClassModule
{
	/** The basic class data. */
	ModClassDescriptor modClassDescriptor;				/* 0 */

	/** Pump function for the module. */
	void (*pProcessFunc)(void *pInstance);				/* 8 */

	/** Bypass function for the module. */
	void (*pBypassFunc)(void *pInstance);				/* 12 */

	/** Set function. */
	UINT32 (*pSet)(void *pInstance, UINT32 mask);		/* 16 */

	/** Get function. */
	UINT32 (*pGet)(void *pInstance, UINT32 mask);		/* 20 */

	/** Module version information */
	UINT32 nModuleVersion;								/* 24 */

	/** The number of construction module parameters. */
	UINT32 nPackedParameters;							/* 28 */

	/** Vector of 64 bits specifying the type of constructor
	arguments. Bit 0 starts at the first member past the
	instance object. Bit set means argument is float.
	Bits past the last argument are don't care, should be 0. */
	UINT32 bitVectorFloat[2];							/* 32 */
}
ModClassModule;		/* 40 bytes */

#ifdef BUILD64

/** Table of offsets for modules. */
extern const UINT32 g_module_offsets[];

/** Table of offsets for module instances. */
extern const UINT32 g_module_instance_offsets[];

/** Table of offsets for wires. */
extern const UINT32 g_wire_offsets[];

/** Dummy for computing 64 bit offsets without having the actual module class available. */
/*private*/ typedef struct _ModClassModule2
{
	ModClassModule module;
	UINT32 m_offset[1];
} ModClassModule2;
#endif

/** Set the packed argument count nPackedParameters. */
#define ClassModule_PackArgCounts(nPublicArgs, nPrivateArgs) \
	(((nPrivateArgs) << 16) | ((nPublicArgs) & 0xffffU))

/** Get the number of public arguments from nPackedParameters. */
#define ClassMod_GetNPublicArgs(packed) \
	((packed) & 0xffffU)

/** Get the number of private arguments from nPackedParameters. */
#define ClassMod_GetNPrivateArgs(packed) \
	(((packed) >> 16) & 0xffffU)


/* This macro declares a module class. For 64 bits we have an extra member
 * which is a vector of member offsets to translate indexes to struct offsets. */

#define PASTE(x, y)		x ## y
#define EVAL(x, y)		PASTE(x, y)

#ifdef BUILD64
/* 64 bit version has an array of member offsets following. */
#define CREATE_MODULE_CLASS(class_name, nMembers)	\
	typedef struct EVAL(_, class_name) \
	{ \
		ModClassModule modBaseClassDescriptor; \
		UINT32 m_offset[(nMembers <= 0)? 1: nMembers]; \
	} class_name;

#else
/* 32 bit version is just an alias for the module class. */
#define CREATE_MODULE_CLASS(class_name, nMembers)	\
	typedef struct EVAL(_, class_name) \
	{ \
		ModClassModule modBaseClassDescriptor; \
	} class_name;

#endif


/** A module instance descriptor. Common to all module instances. */
/*private*/ typedef struct _ModInstanceDescriptor
{
	/** Chain instances together in a linked list. */
	struct _ModInstanceDescriptor *pNextInstance;		/* 0 */

	/** Pointer back to the static descriptor from which this instance was
	created. */
	const ModClassModule *pModClassDescriptor;			/* 4 */

	/** The unique instance ID of this instance. */
	UINT32 nUniqueInstanceID;							/* 8 */
}
ModInstanceDescriptor;		/* 12 bytes */


/** The table of module descriptions. Truly global. */
extern const ModClassModule *g_module_descriptor_table[];

/** The number of elements in g_module_descriptor_table. Truly global. */
extern UINT32 g_module_descriptor_table_size;


/* Low order 2 bits of flags high byte are the module active state. */

/** Mask for active bits. */
#define MODULE_ACTIVE_MASK			0x03000000U

/** Module is active. */
#define MODULE_ACTIVE				0x00000000U

/** Module is bypassed. */
#define MODULE_BYPASS				0x01000000U

/** Module is muted. */
#define MODULE_MUTE					0x02000000U

/** Module is inactive. */
#define MODULE_INACTIVE				0x03000000U


/** This bit allows for deferred execution of the module's set function in the main thread. */
#define MODULE_GENERAL_DEFERRED_SET			0x08000000U

/** Test if the deferred bit is set. */
#define awe_modIsDeferredSet(S)				((((S)->instance.packedFlags) & (MODULE_GENERAL_DEFERRED_SET)) != (0))

/** Set the deferred bit to v */
//#define awe_modSetDeferredSet(S, v)			(S)->instance.packedFlags &= ~MODULE_GENERAL_DEFERRED_SET; (S)->instance.packedFlags |= ((v & 1U) << 27); g_pAWE_instance->m_general_flags |= 0x80
void awe_modSetDeferredSet(void *S, UINT32 v);


/** High 4 bits of flags high byte are available for any use. */
#define MODULE_GENERAL_MASK			0xf0000000U

/** Get the general flag bits. */
#define GetGeneralFlagBits(S)		((((S)->instance.packedFlags) >> (28)) & (0x1fU))

/** Set the genera; flag bits to v. */
#define SetGeneralFlagBits(S, v)	(S)->instance.packedFlags &= ~MODULE_GENERAL_MASK; (S)->instance.packedFlags |= ((v & 0xfU) << 28)


/** A module instance. The instance is variable length on the heap, the size
being determined by the constructor arguments. */
/*private*/ typedef struct _ModuleInstanceDescriptor
{
	/** The basic instance data. */
	ModInstanceDescriptor instanceDescriptor;			/* 0 */

	/** Pointer to owning layout instance (type is a forward reference). */
	struct _LayoutInstance *pOwner;						/* 12 */

	/** Array of input and output wires. There are nInwires + nOutWires + nScratchWires elements. */
	WireInstance **pWires;								/* 16 */

	/** Pump function for the module. May be empty, pProcessFunc, or, pBypassFunc. */
	void (*pProcessFunc)(void *pInstance);				/* 20 */

	/** Control flags. */
	UINT32 packedFlags;									/* 24 */

	/** Module specific profiling time.	 This is number of cycles times 256. */
	UINT32 profileTime;									/* 28 */
}
ModuleInstanceDescriptor;		/* 32 bytes */

/** Call a module's set function. */
INT32 ClassModule_Set(ModuleInstanceDescriptor *pInstance, UINT32 mask);

/** Call a module's get function. */
#define ClassModule_Get(pInstance, mask) \
if ((pInstance) != NULL) { \
	UINT32 (*pGet)(void *a, UINT32 b) = (pInstance)->instanceDescriptor.pModClassDescriptor->pGet; \
	if (pGet != NULL) { \
		(void)pGet(pInstance, mask); \
	} \
}

/** Pack the sizes into the flags word. */
#define ClassModule_PackFlags(nInWires, nOutWires, nScratchWires) \
	((((nScratchWires) & 0xffU) << 16) | (((nOutWires) & 0xffU) << 8) | ((nInWires) & 0xffU))

/** Get the input wire count 0..255 */
#define ClassMod_GetNInWires(packed) \
	((packed) & 0xffU)

/** Get the output wire count 0..255 */
#define ClassMod_GetNOutWires(packed) \
	(((packed) >> 8) & 0xffU)

/** Get the scratch wire count. */
#define ClassMod_GetNScratchWires(packed) \
	(((packed) >> 16) & 0xffU)

/** Get the total number of wires */
#define ClassMod_GetWireCount(packed) \
	(((packed) & 0xffU) + (((packed) >> 8) & 0xffU) + (((packed) >> 16) & 0xffU))

/** Get the module state. */
#define ClassMod_GetModuleState(packed) \
	(((packed) >> 24) & 0x03U)


/** These macros operate on high-level module objects. The goal is to hide the
	details of the implementation from the module writer. */

/** Get the wires that are attached to the module */
#define ClassModule_GetWires(S)		((S)->instance.pWires)

/** Get the input wire count 0..255 */
#define ClassModule_GetNInWires(S) \
	(ClassMod_GetNInWires((S)->instance.packedFlags))

/** Get the output wire count 0..255 */
#define ClassModule_GetNOutWires(S) \
	(ClassMod_GetNOutWires((S)->instance.packedFlags))

/** Get the scratch wire count. */
#define ClassModule_GetNScratchWires(S) \
	(ClassMod_GetNScratchWires((S)->instance.packedFlags))

/** Get the module state. */
#define ClassModule_GetModuleState(S) \
	(ClassMod_GetModuleState((S)->instance.packedFlags))

/** Get the module objectID. */
#define ClassModule_GetObjectID(S) \
	((S)->instance.instanceDescriptor.nUniqueInstanceID)

/*------------------ Layout declarations ------------------*/

/** A layout instance. The instance is variable length on the heap, the size
being determined by the constructor arguments. */
/*private*/ typedef struct _LayoutInstance
{
	/** The basic instance data. */
	InstanceDescriptor instanceDescriptor;

	struct _LayoutInstance *pNextLayout;

	/** When 1, called on every iteration, otherwise called less frequently. */
	UINT32 nDivider;

	/** When zero, will execute, and be set to nDivider-1, otherwise will be decremeneted. */
	UINT32 nCurrentDivide;

	/** Average cycles per process. Times 256. */
	UINT32 averageCycles;

	/** Peak cycles per process. Times 256. */
	UINT32 peakCycles;

	/** Number of processes. */
	UINT32 processCount;

	/** Instantaneous cycles for the entire layout that executed.  Times 256. */
	UINT32 instCycles;

	/** Averaged time in cycles between calls to the processing function.  Times 256. */
	UINT32 timePerProcess;

	/** Percentage CPU time */
	float percentCPU;	// strictly float, not FLOAT32

	/** Number of modules in the layout. */
	UINT32 nModules;

	/** The starting time for the last call to Layout_Process. */
	UINT32 m_last_start_time;

	/* Private. */
	void *m_private;

	/* Private. */
	void *m_private2;

	/** Variable length array of module pointers. */
	ModuleInstanceDescriptor *pModuleInstances[1];
}
LayoutInstance;

#define ALLOCATION_BLOCKSIZE_DWORDS 16
#define MAX_FILENAME_LENGTH_IN_DWORDS 14

#ifndef DIRECTORY_ENTRY_DEFINED
#define DIRECTORY_ENTRY_DEFINED

/**
 * @brief Flash file system directory entry data structure.
 *
 * All structure elements are 4 byte words to accommodate SHARC processors
 *
 * File Info UINT32 layout
 *
 * Attribute bits | File Data CRC | Block Offset to Data
 * -------------- | ------------- | --------------------
 * 1 byte         | 1 byte        |  2 bytes
 *
 * | bit 31......................................bit 0 |
 *
 *
 * Attribute              | Value
 * ---------------------- | -----
 * LOAD_IMAGE			  | 0x01
 * STARTUP_FILE			  | 0x02
 * DATA_FILE		      | 0x04
 * COMPILED_SCRIPT	      | 0x08
 * COMMAND_SCRIPT	      |	0x10
 * PRESET_SCRIPT	      |	0x20
 * COMPILED_PRESET_SCRIPT |	0x28
 * LOADER_FILE			  | 0x40
 * FILE_DELETED			  | 0x80
 *
 *
 * The layout for nFileInfo field in a file directory entry is:
 * ---------------------------------------------------------
 * | Attribute bits | File Data CRC | Block Offset to Data |
 * ---------------------------------------------------------
 * | 1 byte			| 1 byte		| 2 bytes			   |
 * ---------------------------------------------------------
 * | bit 31...........................................bit 0|
 * ---------------------------------------------------------
 *
 * Note that the file name must fit in one allocation block
 * This inherently limits the file name to 55 bytes plus zero terminator byte (56 bytes)
 */
typedef struct _DIRECTORY_ENTRY
{
	UINT32 nFileInfo;
	UINT32 nDataDWordCnt;
    UINT32 nFilename[MAX_FILENAME_LENGTH_IN_DWORDS];
} DIRECTORY_ENTRY, *PDIRECTORY_ENTRY;
#endif

/**
 * All structure elements are 4-byte DWords
 *
 * Flash Memory Layout:
 *
 * TOP OF FLASH MEMORY
 *
 * File Header Info
 * Allocation table bit map
 * File Directory
 * File data
 * Application Code
 * Boot Loader
 *
 * BOTTOM OF FLASH MEMORY
 *
 * File System Header layout (padded with zeros to fill one allocation block)
 *-------------------------------------------------------------------------------------------------------
 *| Version Number | Start of Data Addr | File Directory Addr | Alloc Table Addr | Allocation Block Size |
 *-------------------------------------------------------------------------------------------------------
 *| 4 bytes        | 4 bytes            | 4 bytes             | 4 bytes          | 4 bytes               |
 *-------------------------------------------------------------------------------------------------------
 *
 * EXAMPLE Allocation Table
 *
 * Assume there are 524288 bytes available in the high end of the flash address space.
 * This is the same as 131072 4-byte DWords.
 * 131072 words divided by 16 words (allocation block size) = 8192 allocation units.
 * There are 32 bits per bitmap word so there needs to be 8192 / 32  = 256 words
 * allocated to the bit map if the allocation table is mapping the entire memory
*/

typedef struct _FSAttributes
{
    /** Byte Address in flash memory of start of file allocation table */
    UINT32 nAllocTableBitMapFlashAddr;

    /** Block size for the file system 16 words/block */
    UINT32 nAllocBlockSizeInDWords;

    /** Block size for the file system 64 bytes/block */
    UINT32 nAllocBlockSizeInBytes;

    /** Byte Address in flash memory of the first entry of the file directory */
    UINT32 nFileDirectoryFlashAddr;

    /** TOtal flash memory size in bytes */
    UINT32 nFlashMemorySizeBytes;

    /** Size of flash memory eraseable block size in bytes */
    UINT32 nEraseableBlockSize;

    /** Count of active files saved in the flash file system */
    INT32  nFileCnt;

    /** Byte Address in flash memory of the start of file data */
    UINT32 nStartOfFileDataFlashAddr;

    /** Count of the number of file allocation blocks in use */
    UINT32 nDataBlocksInUse;

    /** Current directory entry when navigating through the file system */
    DIRECTORY_ENTRY CurrentDirEntry;

    /** Address of first free block for file allocation */
    UINT32 nFirstFreeBlockFlashAddr;

    /** File is open state */
    BOOL   bFileOpen;

    /** Number of DWords read so far */
    INT32  nDWordsRead;

    /** Number of DWords written so far */
    INT32  nDWordsWritten;

    /** Current file position byte offset */
    UINT32 nFileCurrentPosByteOffset;

    /** Offset to start of file content area in flash */
    UINT32 nFileContentReadOffset;

    /** File attribute byte */
    UINT32 nNewFileAttributeByte;

    /** Current directry entry when navigating */
    UINT32 CurrentDirEntryFlashAddr;

    /** Start of chain of deleted files */
    UINT32 nFileDirectoryFlashAddr_toDel;

    /** Residue DWords */
    UINT32 ResidueDWords[ALLOCATION_BLOCKSIZE_DWORDS];

    /** Number of DWords remaining to write */
    UINT32 nRemainingDWordsToWrite;

    /** Where in flash memory the flash file system starts */
    UINT32 FlashStartOffset;

    /** Flash file system information */
    FileSystemInfo filesystem_info;

} FSAttributes;

typedef struct _AWEFlashFSInstance
{
    /** Size of flash memory - if non-zero, next two values must also be non-zero. */
	UINT32 flashMemorySizeInBytes;

	/** Size of flash erase block. */
	UINT32 flashErasableBlocksizeInBytes;

	/** Offset into start of flash used for file system. */
	UINT32 flashFileSystemStartOffsetInBytes;

    /** Flash erase time in milliseconds */
    UINT32 flashEraseTimeInMs;

    /** User function to initialize flash file system, */
    BOOL (*initFlashFileSystemCallback)(void);

    /** User function to erase one or more sectors. */
    BOOL (*eraseFlashMemorySectorCallback)(UINT32 nStartingAddress, UINT32 nNumberOfSectors);

    /** User function to write to flash. */
    BOOL (*writeFlashMemoryCallback)(UINT32 nFlashAddress, UINT32 * pBuffer, UINT32 nDWordsToWrite);

    /** User function to read from flash. */
    BOOL (*readFlashMemoryCallback)(UINT32 nFlashAddress, UINT32 * pBuffer, UINT32 nDWordsToRead);

    FSAttributes flashAttributes;

} AWEFlashFSInstance;


/** The AWE instance object. */
/*private*/ typedef struct _AWEInstance
{
	/** The ID of this instance. */
	UINT32 m_coreID;

	/** The master heap. */
	UINT32 *m_master_heap;

	/** The second fast heap. */
	UINT32 *m_fastb_heap;

	/** The slow heap. */
	UINT32 *m_slow_heap;

	/** The fast heap size. */
	UINT32 m_master_heap_size;

	/** The second heap size. */
	UINT32 m_fastb_heap_size;

	/** The slow heap size. */
	UINT32 m_slow_heap_size;

	/** Point to BSP audio start. */
	INT32 (*m_pAwe_pltAudioStart)(struct _AWEInstance *pAWE);

	/** Point to BSP audio stop. */
	INT32 (*m_pAwe_pltAudioStop)(struct _AWEInstance *pAWE);

	/** Point to BSP input pin table. */
	IOPinDescriptor *m_pInterleavedInputPin;

	/** Point to BSP output pin table. */
	IOPinDescriptor *m_pInterleavedOutputPin;

	/** Number of modules in module table. */
	UINT32 m_module_descriptor_table_size;

	/** Pointer to module table. */
	const ModClassModule **m_pModule_descriptor_table;

    /** Packet buffer pointer. */
    UINT32 *pPacketBuffer;

	/** Optional reply buffer pointer. */
	UINT32 *pReplyBuffer;

    /** Packet buffer size. */
    UINT32 packetBufferSize;

    /** User Version word */
    UINT32 userVersion;

	/** Core speed in Hz. */
    float coreSpeed;

	/** Profiling clock speed in Hz. */
    float profileSpeed;

#if defined(__ADSP21000__) && !defined(__ADSPSC589_FAMILY__)
    /** Name of this instance */
    const UINT32 *processorName;
#else
    /** Name of this instance */
    const char *processorName;
#endif

	/** Number of threads supported (1-4). */
    UINT32 numThreads;

	/** Default sample rate of this instance. */
    float sampleRate;

	/** Base frame size of this instance. */
    UINT32 fundamentalBlockSize;

    /** Flash file system instance */
    AWEFlashFSInstance * pFlashFileSystem;

// Private members.

	/** Head of chain of class instances. */
	InstanceDescriptor *m_pInstanceHead; 

	/** Tail of chain of class instances. */
	InstanceDescriptor *m_pInstanceTail;

	/** Head of chain of I/O pin instances.
	Must be initialized to point at the last item in the above. */
	IOPinDescriptor *m_pIOPinHead;

	/** Tick instance pointer. */
	InstanceDescriptor *m_pTickObject;

	/** Head of list of layouts. */
	LayoutInstance *m_pLayoutHead;

	/** Tail of list of layouts. */
	LayoutInstance *m_pLayoutTail;

	/** The pointer to the lookup table - initially NULL. */
	void **m_lookup_table;

	/** Index of next free second heap word. */
	UINT32 m_fastb_heap_index;

	/** Index of next free master heap word. */
	UINT32 m_master_heap_index;

	/** Index of next free slow heap word. */
	UINT32 m_slow_heap_index;

	/** Initialized to 0 on program start or restart, incremented each time an
	instance of anything is created. */
	UINT32 m_unique_instance_ID_counter;

	/** The maximum ID supported by the lookup table. */
	UINT32 m_maxID;

	/** Some general purpose flag bits. */
	UINT32 m_general_flags;

	/** Head of linked list of bound wires. */
	WireInstance *m_pBindHead;

	/** Previous sequence number for AWE packets. */
	UINT32 msg_seglast;

	/** Extended target info. */
	ExtendedInfo extendedInfo;

	/** Size of inputPin table array, default 1. */
    UINT32 inputPinCount;

	/** Size of outputPin table array, default 1. */
    UINT32 outputPinCount;

	/** Type of this instance (ARM9, CortexA, long set of possible values). */
    INT32 processorType;

	/** TRUE if target is float. */
    BOOL hasFloatSupport;

	/** TRUE if the instance is SMP. */
    BOOL isSMP;

// Private function pointers.

    /** Platform init flash file system */
    BOOL (*m_pAwe_pltInitFlashFileSystem)(AWEFlashFSInstance * pAWEFlashFSInstance);

    /** Platform execute file files */
    INT32 (*m_pAwe_pltExecuteFlashFiles)(struct _AWEInstance * pAWE, AWEFlashFSInstance * pAWEFlashFSInstance);

    /** Platform flash file system command handler */
    INT32 (*m_pAwe_pltFlashCommandHandler)(struct _AWEInstance * pAWE);

	/** A function pointer for instance packet process. */
//	INT32 (*m_pAwe_fwPacketProcess)(struct _AWEInstance *pAWE);

} AWEInstance;

/**
 * @brief Get the size of the private AWE instance object.
 * @return							its size
 */
UINT32 awe_fwSizeOfInstance(void);

/**
 * @brief Get the size of the public AWE instance object.
 * @return							its size
 */
UINT32 awe_pltSizeOfInstance(void);

/**
 * @brief Initialize a pin object.
 * @param [in] pPin					the pin instance to initialize
 * @param [in] channels				number of channels
 * @param [in] name					optional pin name
 * @return							0 or error code
 */
INT32 awe_initPin(IOPinDescriptor *pPin, UINT32 channels, const char *name);

/** Describe an arbitrary core. */
/*private*/ typedef struct _CoreDescriptor
{
	/**
	 * @brief Constructor.
	 * @param [in] pData		pointer to this object
	 * @param [in] pClasses		pointer to class table or NULL
	 * @param [in] nClasses		number of classes
	 * @param [in] pExtra		commands pointer
	 */
	int (*m_pConstructor)(void *pData, void *pClasses, UINT32 nClasses, void *pExtra);

	/**
	 * @brief Pointer to post constructor - might need to modify IPC pins.
	 * @param [in] pData		pointer to this CoreDescriptor
	 * @param [in] pIpcPins		pointer to table of IPC pins
	 * @param [in] nIpcPins		size of the table
	 */
	int (*m_pPostConstructor)(void *pData, void *pIpcPins, UINT32 nIpcPins);

	/**
	 * @brief Destructor.
	 * @param [in] pData		pointer to this CoreDescriptor
	 */
	int (*m_pDestructor)(void *pData);

	/**
	 * @brief Pointer to core specific pump function.
	 * @param [in] pData		pointer to this CoreDescriptor
	 */
	void (*m_pPumpCore)(void *pData);

	/**
	 * @brief Pointer to core specific command function.
	 * @param [in] pData		pointer to this CoreDescriptor
	 * @param [in] txBuffer		send packet buffer
	 * @param [out] rxBuffer	receive packet buffer
	 */
	INT32 (*m_pSendCommand)(void *pData, UINT32 *txBuffer, UINT32 *rxBuffer);

	/**
	 * @brief Pointer to core write wire function.
	 * @param [in] pData		pointer to this CoreDescriptor
	 * @param [in] pWire		poinyter to core wire in core address space
	 * @param [in] nWords		number of words to copy
	 * @param [in] pWords		pointer to words to write
	 */
	void (*m_pWriteInputWire)(void *pData, WireInstance *pWire, UINT32 nWords, Sample *pWords);

	/**
	 * @brief Pointer to core read wire function.
	 * @param [in] pData		pointer to this CoreDescriptor
	 * @param [in] pWire		poinyter to core wire in core address space
	 * @param [in] nWords		number of words to copy
	 * @param [out] pWords		pointer to words to read
	 */
	void (*m_pReadOutputWire)(void *pData, WireInstance *pWire, UINT32 nWords, Sample *pWords);

	/**
	 * @brief Pointer to core get samples function..
	 * @param [in] pData		pointer to this CoreDescriptor
	 * @param [in] pWire		poinyter to core wire in core address space
	 * @return					number of samples in core wire
	 */
	int (*m_GetNumSamples)(void *pData, WireInstance *pWire);

	/**
	 * @brief Pointer to core get channels function.
	 * @param [in] pData		pointer to this CoreDescriptor
	 * @param [in] pWire		poinyter to core wire in core address space
	 * @return					number of samples in core wire
	 */
	int (*m_GetNumChannels)(void *pData, WireInstance *pWire);

	/**
	 * @brief get the AWE instance
	 * @param [in] pData		pointer to this CoreDescriptor
	 * @return					instance pointer or NULL if in different address space
	 */
	AWEInstance *(*m_GetInstance)(void *pData);

	/** Proxy to IAudioCommands::ReadAndCopyFromInput. */
	int (*m_pReadAndCopy)(void *pCmds, int *pSamples, UINT32 nOutputChans, UINT32 reqSamples, void *pvDevice, UINT32 nInputChans, UINT32 *pchIdx, UINT32 pumpCount);

	/** Proxy to CWaveWriter::WriteSamples. */
	int (*m_pWriteSamples)(void *pvWriter, void *pSamples, UINT32 samples);

	/** Generic data. */
	void *m_pData;

	/** Which core this is. */
	UINT32 m_coreID;

	/** Set for SMP cores. */
	UINT32 m_isSMP;
} CoreDescriptor;

/* Notification constants for owned pins. */

#define AWE_NOTIFY_BIND		0U		// BSP pin is bound

#define AWE_NOTIFY_DESTROY	1U		// Target is destroyed, so are all pin bindings

/** Set when core is SMP. */
#define CORE_IS_SMP				1U

/** Set when core is self-pumping (can't be SMP). */
#define CORE_IS_SELF_PUMPED		2U

/** Test if a core is SMP. */
#define GetCoreIsSMP(x) \
	((x)->m_isSMP & CORE_IS_SMP)

/** Test if a core is self-pumping. */
#define GetCoreSelfPumped(x) \
	((x)->m_isSMP & CORE_IS_SELF_PUMPED)

/** Thread structure to record times and core pump completion. */
/*private*/ typedef struct _SRunPump
{
	/** The core ID this is pumping for. */
	int m_core;

	/** Which layout to pump. */
	int m_layoutIndex;

	/** The AWE instance we are calling. */
	void *m_pData;

	/** Utility pointer. */
	void *m_pData2;

	/** Passed in argument. */
	UINT32 m_nPumpSamples;

	/** Passed in argument. */
	UINT32 m_pumpCount;

	/** Average cycles for this core. */
	UINT32 m_averageCycles;

	/** Total time for this core. */
	UINT32 m_timePerProcess;

	/** The result - negative if error. */
	volatile int m_result;

	/** More thread working detection. */
	volatile int m_in_pump_count;

	/** Tick number when thread must finish. */
	UINT32 m_deadline;
} SRunPump;

/**
 * @brief Get the core descriptor for the given core.
 * param [in] pOwner					core object owner
 * @param [in] coreID					ID of core to get
 * @return								core descriptor
 */
CoreDescriptor *awe_pltGetCore(void *pOwner, UINT32 coreID);

/** Bit to record if the tick (idle) function was called. */
#define AWE_TICKED			0x01U

/** Bit to suppress profiling. */
#define AWE_NO_PROFILE		0x10U

/** Bit to suppress outer profiling. */
#define AWE_NO_PROFILE2		0x20U

/** Audio started flag. */
#define AWE_AUDIO_STARTED	0x40U

/** A deferred bit was set. */
#define AWE_DEFERRED_SET	0x80U

//////////////////////////////////////////////////////////AP NEW/////////////////////////////////////////////////////
/** Layout Loaded flag. */
#define AWE_LAYOUT_VALID	0x100U //

/** Set the layout loaded flag. */
#define SET_LAYOUT_VALID(pAWE)	((pAWE)->m_general_flags |= AWE_LAYOUT_VALID)

/** Clear the started flag. */
#define CLR_LAYOUT_VALID(pAWE)	((pAWE)->m_general_flags &= ~AWE_LAYOUT_VALID)

/** Is audio started. */
#define IS_LAYOUT_VALID(pAWE)	(((pAWE)->m_general_flags & AWE_LAYOUT_VALID) > 0 ? 1:0)

//////////////////////////////////////////////////////AP END NEW ///////////////////////////////////////////////////////

/** Set the started flag. */
#define SET_STARTED(pAWE)	((pAWE)->m_general_flags |= AWE_AUDIO_STARTED)

/** Clear the started flag. */
#define CLR_STARTED(pAWE)	((pAWE)->m_general_flags &= ~AWE_AUDIO_STARTED)

/** Is audio started. */
#define IS_STARTED(pAWE)	(((pAWE)->m_general_flags & AWE_AUDIO_STARTED) > 0 ? 1:0)

/** Is deferred bit set. */
#define IS_DEFERRED_SET(pAWE)	((pAWE)->m_general_flags & AWE_DEFERRED_SET)

/** Clear the deferred bit. */
#define CLR_DEFERRED_SET(pAWE)	(pAWE)->m_general_flags &= ~AWE_DEFERRED_SET

/** Used in special modules that need to access the framework. */
extern AWEInstance *g_pAWE_instance;

#ifdef USE_AFLOAT
/**
 * @brief Convert Knowles float to IEEE float.
 * @param val			Knowles value to convert
 * @return				IEEE value
 */
float awe_fwConvertToIEEE(fr32 val);

/**
 * @brief Convert an IEEE float to Knowles float.
 * @param val			IEEE value to convert
 * @return				Knowles value
 */
fr32 awe_fwConvertFromIEEE(float val);

#endif

/**
 * @brief Initialize the instance's target info.
 * @brief [out] pTargetInfo		-  target info to initialize
 * @param [in] coreID           -  Core ID: must be 0 (increments sequentially for multi core systems)
 * @param [in] coreClockSpeed   -  Processor core clock speed in Hz
 * @param [in] sampleSpeed      -  Profiling clock - can be same as core clock or smaller than core clock. Most embedded targets will have the same value as core clock
 * @param [in] name             -  String for name of the target up to 16 characters maximum (Just for the display)
 * @param [in] procType         -  Type of the processor (just for display).
 * @param [in] isFloat          -  Native type of the processor - floating or fixed (1 - for floating point, 0 - for fixed point)
 * @param [in] isFlash          -  AWE flash file system supported or not (1 - for supported and 0 - for not supported)
 * @param [in] nInputs          -  Number of input pins
 * @param [in] nOutputs         -  Number of output pins
 * @param [in] isSMP            -  Is the processor of type static core or SMP (1 - SMP, otherwise static)
 * @param [in] threads          -  Number of threads supported. Default every AWE target will have 2 threads to support two different block processing
 * @param [in] sampleRate       -  Fundamental sample rate of the system (default to 48KHz)
 * @param [in] nInputChannels   -  Number of input channels
 * @param [in] nOutputChannels  -  Number of output channels
 * @param [in] nVerDay          -  Day number used in version string, 6.Day.Month.Year (6 is the AWE version)
 * @param [in] nVerMonth        -  Month number used in version string, 6.Day.Month.Year
 * @param [in] nVerYear         -  Year number used in version string, 6.Day.Month.Year
 * @param [in] blockSize        -  Audio Weaver fundamental block - 32 in provided BSP (32 is the default used in many targets)
 * @param [in] commandBufferLen -  Communication buffer length in 32-bit words
 * @param [in] nCores			-  Number of cores
 */
/*private*/ void awe_fwInitTargetInfo(TargetInfo *pTargetInfo, UINT32 coreID, float coreClockSpeed, float sampleSpeed,
						  const char *name, INT32 procType, INT32 isFloat, INT32 isFlash,
						  INT32 nInputs, INT32 nOutputs, INT32 isSMP, INT32 threads, float sampleRate,
						  INT32 nInputChannels, INT32 nOutputChannels, INT32 nVerDay, INT32 nVerMonth, INT32 nVerYear, UINT32 blockSize, UINT32 commandBufferLen);

/**
 * @brief Initialize the packed name (not SHARC).
 * @param [in] pPackedName			pointer to packed name array
 * @param [in] name					name as a string
 */
/*private*/ void awe_fwSetPackedName(UINT32 *pPackedName, const char *name);

/**
 * @brief Given an address in {objectID,index} form, compute the
 * address of the module member.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] address				the member index
 * @param [out] retVal				0 or error code
 * @return							pointer to member
 */
/*private*/ UINT32 *awe_fwAddressOfMember(const AWEInstance *pAWE, UINT32 address, INT32 *retVal);

/**
 * @brief Given an address in {objectID,index} form, compute the
 * address of the module member. Also return the module pointer.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] address				the member index
 * @param [out] pMod				pointer to receive module pointer
 * @param [out] retVal				0 or error code
 * @return							pointer to member
 */
/*private*/ UINT32 *awe_fwAddressOfMember2(const AWEInstance *pAWE, UINT32 address, ModuleInstanceDescriptor **pMod, INT32 *retVal);

/**
 * @brief Dereference a pointer to a nested module
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] address				address in 20.12 {objectID,index} form
 * @param [out] retVal				0 or error code
 * @return							ID of object pointed to, 0 on error
 */
/*private*/ INT32 awe_fwDerefPointer(const AWEInstance *pAWE, UINT32 address, INT32 *retVal);

/**
 * @brief Assign a real pointer to a pointer type left hand side.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] addressLhs			LHS address in 20.12 {objectID,index} form
 * @param [in] addressRhs			RHS address in 20.12 {objectID,index} form
 * @param [in] offsRhs				offset of right hand side
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwSetPointer(const AWEInstance *pAWE, INT32 addressLhs, INT32 addressRhs, UINT32 offsRhs);

/**
 * @brief Change the ID of a module
 * @param [in] moduleID				ID of module to change
 * @param [in] newModuleID			ID to assign
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwSetInstanceID(const AWEInstance *pAWE, UINT32 moduleID, UINT32 newModuleID);

/**
 * Fetch the value at the given address.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] address				address in 20.12 {objectID,index} form
 * @param [in] floatFlg				value is float
 * @param [out] retVal				0 or error code
 * @return							fetched value, 0 on error
 */
/*private*/ INT32 awe_fwFetchValue(const AWEInstance *pAWE, UINT32 address, INT32 *retVal, UINT32 ptrOffset, UINT32 floatFlg);

/**
 * @brief Fetch the value at the given address after get call.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] address				address in 20.12 {objectID,index} form
 * @param [in] mask					get call mask
 * @param [in] floatFlg				value is float
 * @param [out] retVal				0 or error code
 * @return							fetched value, 0 on error
 */
/*private*/ INT32 awe_fwGetCallFetchValue(const AWEInstance *pAWE, UINT32 address, UINT32 mask, INT32 *retVal, UINT32 ptrOffset, UINT32 floatFlg);

/**
 * @brief Set the value at the given address.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] address				address in 20.12 {objectID,index} form
 * @param [in] floatFlg				value is float
 * @param [in] value				value to set
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwSetValueDirect(const AWEInstance *pAWE, UINT32 address, INT32 value, UINT32 ptrOffset, UINT32 floatFlg);

/**
 * @brief Set the value at the given address, then do set call.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] address				address in 20.12 {objectID,index} form
 * @param [in] value				value to set
 * @param [in] mask					set call mask
 * @param [in] floatFlg				value is float
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwSetValueSetCall(const AWEInstance *pAWE, UINT32 address, INT32 value, UINT32 mask, UINT32 ptrOffset, UINT32 floatFlg);

/**
 * @brief Fetch the values at the given address.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] address				address in 20.12 {objectID,index} form
 * @param [in] offset				array offset
 * @param [in] argSize				number of arguments
 * @param [out] args				returned values
 * @param [in] floatFlg				when set, values are float
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwFetchValues(const AWEInstance *pAWE, UINT32 address, UINT32 offset, UINT32 argSize, Sample *args, UINT32 floatFlg);

/**
 * Fetch the values at the given address after get call.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] address				address in 20.12 {objectID,index} form
 * @param [in] offset				array offset
 * @param [in] mask					get call mask
 * @param [in] argSize				number of arguments
 * @param [out] args				returned values
 * @param [in] floatFlg				float values when set
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwGetCallFetchValues(const AWEInstance *pAWE, UINT32 address, UINT32 offset, UINT32 mask, UINT32 argSize, Sample *args, UINT32 floatFlg);

/**
 * @brief Write the arguments to the given address.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] address				address in 20.12 {objectID,index} form
 * @param [in] offset				array offset
 * @param [in] argSize				number of arguments
 * @param [in] args					values to set
 * @param [in] floatFlg				float values when set
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwSetValues(const AWEInstance *pAWE, UINT32 address, UINT32 offset, UINT32 argCount, const Sample *args, UINT32 floatFlg);

/**
 * @brief Write the arguments to the given address, then set call.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] address				address in 20.12 {objectID,index} form
 * @param [in] offset				array offset
 * @param [in] mask					set call mask
 * @param [in] argSize				number of arguments
 * @param [in] args					values to set
 * @param [in] floatFlg				float values when set
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwSetValuesSetCall(const AWEInstance *pAWE, UINT32 address, UINT32 offset, UINT32 mask, UINT32 argCount, const Sample *args, UINT32 floatFlg);

/**
 * @brief Returns the number of channels in the Layout's primary input and output pins.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] pinIdx				which input pin
 * @param [out] inCount				channels in primary input pin
 * @param [out] outCount			channels in primary output pin
 */
/*private*/ void awe_layoutGetChannelCount(const AWEInstance *pAWE, UINT32 pinIdx, UINT32 *inCount, UINT32 *outCount);

/**
 * @brief Returns a pointer to where the platform should write input data.
 * @param [in] pinIdx				which input pin
 * @param [in] chan					which channel
 * @param [out] stride				data stride
 * @return							write pointr or NULL on error
 */
/*private*/ INT32 *awe_fwGetInputChannelPtr(const AWEInstance *pAWE, UINT32 pinIdx, UINT32 chan, INT32 *stride);

/**
 * @brief Returns a pointer to where the platform should read output data from.
 * @param [in] pinIdx				which input pin
 * @param [in] chan					which channel
 * @param [out] stride				data stride
 * @return							read pointr or NULL on error
 */
/*private*/ INT32 *awe_fwGetOutputChannelPtr(const AWEInstance *pAWE, UINT32 pinIdx, UINT32 chan, INT32 *stride);

/**
 * @brief Returns the block size of an input pin.
 * @param [in] pinIdx				which input pin
 * @return							blocks size or 0 on error
 */
/*private*/ INT32 awe_layoutGetInputBlockSize(const AWEInstance *pAWE, UINT32 pinIdx, UINT32 *blockSize);

/**
 * @brief Returns the sample rate of an input pin.
 * @param [in] pinIdx				which input pin
 * @return							blocks size or 0 on error
 */
/*private*/ INT32 awe_layoutGetInputSampleRate(const AWEInstance *pAWE, UINT32 pinIdx, FLOAT32 *sampleRate);

/**
 * @brief Handles double buffering of the Layout's I/O pins.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] pinIdx				which input pin
 * #param [in] samplesPerTick		number of samples this tick
 * @return							bitvector of ready layouts
 */
/*private*/ UINT32 awe_fwAudioDMAComplete(const AWEInstance *pAWE, UINT32 pinIdx, INT32 samplesPerTick);

/**
 * @brief Evaluate the layout mask
 * @param [in] pAWE					AWE instance pointer (this)
 * @return							bitvector of ready layouts
 */
/*private*/ UINT32 awe_fwEvalLayoutMask(const AWEInstance *pAWE, UINT32 dividers[4]);

/** Return sizeof INT32 for the target. */
/*private*/ UINT32 awe_fwGetSizeofInt(void);

/**
 * #brief Allocate a block of storage from the heap.
 * @param [in] size					size in bytes of storage to allocate
 * @param [in] heapIndex			which heap(s) to allocate on
 * @param [out] retVal				error code on failure
 * @return							pointer to allocated storage
 *
 * If zero returned, the reason is returned in retVal. The storage is zeroed.
 */
/*private*/ void *awe_fwMalloc(UINT32 size, UINT32 heapIndex, INT32 *retVal);

/**
 * #brief Allocate an aligned block of storage from the heap.
 * @param [in] size					size in bytes of storage to allocate
 * @param [in] align				alignemnt required in bytes
 * @param [in] heapIndex			which heap(s) to allocate on
 * @param [out] retVal				error code on failure
 * @return							pointer to allocated storage
 *
 * If zero returned, the reason is returned in retVal. The storage is zeroed.
 * Alignments are coerced to a multiple of sizeof(int).
 */
/*private*/ void *awe_fwMallocAligned(UINT32 size, UINT32 align, UINT32 heapIndex, INT32 *retVal);

/**
 * #brief Allocate a block of storage from the heap.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] size					size in bytes of storage to allocate
 * @param [in] heapIndex			which heap(s) to allocate on
 * @param [out] retVal				error code on failure
 * @return							pointer to allocated storage
 *
 * If zero returned, the reason is returned in retVal. The storage is zeroed.
 */
/*private*/ void *awe_fwMalloc2(AWEInstance *pAWE, UINT32 size, UINT32 align, UINT32 heapIndex, INT32 *retVal);

/**
 * #brief Allocate a block of storage from shared memory.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] size					size in bytes of storage to allocate
 * @param [out] retVal				error code on failure
 * @return							pointer to allocated storage
 *
 * If zero returned, the reason is returned in retVal.
 * Implemented only for those targets that need special handling
 * which should only br non-SMP multicore systems.
 */
#if !defined(EXTERN_ALLOC_SHARED)
#define awe_pltMallocShared(pAWE, size, retVal) \
	awe_fwMalloc2(pAWE, size, sizeof(INT32), AWE_HEAP_FAST2SLOW, retVal)
#define awe_pltFreeShared(pAWE, buffer)
#else
extern void *awe_pltMallocShared(AWEInstance *pAWE, UINT32 size, UINT32 *retVal);
extern INT32 awe_pltFreeShared(AWEInstance *pAWE, void *buffer);
#endif

/**
 * @brief Return the sizes of the heaps.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [out] heaps				array of 6 words to receive values
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwGetHeapSize(const AWEInstance *pAWE, UINT32 *pHeaps);

/**
 * @brief Reset the framework to its initial state.
 * @param [in] pAWE					AWE instance pointer (this)
 */
/*private*/ void awe_fwDestroy(AWEInstance *pAWE);

/**
 * @brief Create the AWE heaps.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] master_heap_size		requested size of master heap
 * @param [in] slow_heap_size		requested size of slow heap
 * @param [in] fastb_heap_size		requested size of other heap
 * @return							error code
 */
/*private*/ INT32 awe_fwAllocateHeaps(AWEInstance *pAWE, UINT32 master_heap_size, UINT32 slow_heap_size, UINT32 fastb_heap_size);

/**
 * @brief Destroy the heaps allocated by awe_fwAllocateHeaps.
 * @param [in] pAWE					AWE instance pointer (this)
 */
/*private*/ void awe_fwDestroyHeaps(AWEInstance *pAWE);

/**
 * @brief Copy the target info to the caller.
 * @param [out] pTarget				buffer to receive target info
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwGetTargetInfo(const AWEInstance *pAWE, TargetInfo *pTarget);

/**
 * @brief Copy the file system info to the caller.
 * @param [out] pFileSystemInfo		buffer to receive file system info
 * @return							0 or error code
 */
/*private*/ INT32 GetFileSystemInfo(AWEFlashFSInstance * pFS, FileSystemInfo *pFileSystemInfo);

/**
 * @brief Get the next instance number.
 * @param [in] pAWE					AWE instance pointer (this)
 * #return							next instance ID
 */
/*private*/ UINT32 awe_fwGetNextInstanceID(AWEInstance *pAWE);

/**
 * @brief Get the first object in the object chain.
 * @param [out] pObject				pointer to receive first object pointer
 * @param [out] pClassID			pointer to receive first object class
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwGetFirstObject(InstanceDescriptor **pObject, UINT32 *pClassID);

/**
 * @brief Get the first object in the object chain.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [out] pObject				pointer to receive first object pointer
 * @param [out] pClassID			pointer to receive first object class
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwGetFirstObject2(const AWEInstance *pAWE, InstanceDescriptor **pObject, UINT32 *pClassID);

/**
 * @brief Get the next object in the object chain.
 * @param [in] currentObject		current object pointer
 * @param [out] pObject				pointer to receive next object pointer
 * @param [out] pClassID			pointer to receive next object class
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwGetNextObject(const InstanceDescriptor *currentObject, InstanceDescriptor **pObject, UINT32 *pClassID);

/**
 * @brief Get an object based on its unique ID.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] ID					ID of object to find
 * @param [out] pObject				pointer to receive found object pointer
 * @param [out] pClassID			pointer to receive found object class
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwGetObjectByID(const AWEInstance *pAWE, UINT32 ID, InstanceDescriptor **pObject, UINT32 *pClassID);

/**
 * @brief Get an object class from its handle.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] handle					ID of object to find
 * @param [out] pClassID			pointer to receive found object class
 * @return							0 or error code
 */
INT32 awe_fwGetObjectClass(const AWEInstance *pAWE, UINT32 handle, UINT32 *pClassID);

/**
 * @brief Get an object based on its unique ID.
 * @param [in] ID					ID of object to find
 * @param [out] pObject				pointer to receive found object pointer
 * @param [out] pClassID			pointer to receive found object class
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwGetObjectByID2(UINT32 ID, InstanceDescriptor **pObject, UINT32 *pClassID);

/**
 * @brief Get the first I/O object in the object chain.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [out] pObject				pointer to receive first object pointer
 * @param [out] pClassID			pointer to receive first object class
 * @param [out] sampleRate			pin sample rate
 * @param [out] info1				pin info1
 * @param [out] info2				pin info2
 * @param [out] info3				pin info3
 * @param [out] name0				first name word
 * @param [out] name1				second name word
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwGetFirstIO(const AWEInstance *pAWE, InstanceDescriptor **pObject, UINT32 *pClassID, float *sampleRate,
	UINT32 *info1, UINT32 *info2, UINT32 *info3, UINT32 *name0, UINT32 *name1);

/**
 * @brief Get the next I/O object in the object chain.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] pinID				ID of current object
 * @param [out] pObject				pointer to receive next object pointer
 * @param [out] pClassID			pointer to receive next object class
 * @param [out] sampleRate			pin sample rate
 * @param [out] info1				pin info1
 * @param [out] info2				pin info2
 * @param [out] info3				pin info3
 * @param [out] name0				first name word
 * @param [out] name1				second name word
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwGetNextIO(const AWEInstance *pAWE, UINT32 pinID, InstanceDescriptor **pObject, UINT32 *pClassID, float *sampleRate,
	UINT32 *info1, UINT32 *info2, UINT32 *info3, UINT32 *name0, UINT32 *name1);

/**
 * @brief Get the type of a wire.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] wireID				ID of wire to get
 * @param [out] pClassID			class of wire
 * @param [out] sampleRate			sample rate of wire
 * @param [out] info1				first info word
 * @param [out] info2				second info word
 * @param [out] info3				third info word
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwGetWireType(const AWEInstance *pAWE, UINT32 wireID, UINT32 *pClassID, float *sampleRate, UINT32 *info1, UINT32 *info2, UINT32 *info3);

/**
 * @brief Get the type of a pin.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] wireID				ID of pin to get
 * @param [out] pClassID			class of pin
 * @param [out] pBoundID			ID of bound wire
 * @param [out] sampleRate			sample rate of wire
 * @param [out] info1				first info word
 * @param [out] info2				second info word
 * @param [out] info3				third info word
 * @param [out] name0				first name word
 * @param [out] name1				second name word
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwGetPinType(const AWEInstance *pAWE, UINT32 pinID, UINT32 *pClassID, UINT32 *pBoundID, float *sampleRate,
	UINT32 *info1, UINT32 *info2, UINT32 *info3, UINT32 *name0, UINT32 *name1);

/**
 * @brief Return the type of the class this instance was created from.
 * @param [in] pClass				class to look in
 * @return							class ID
 */
/*private*/ UINT32 awe_fwGetClassType(const InstanceDescriptor *pClass);

/**
 * @brief Generic wire instance constructor used by all wire constructors.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [out] retVal				to receive the error code
 * @param [in] sampleRate			sample rate of wire
 * @param [in] info1				packed wire info1
 * @param [in] info2				packed wire info2
 * @param [in] info3				packed wire info3
 * @param [in] pClass				point to wire class
 * @return							constructed object or NULL on error
 */
/*private*/ InstanceDescriptor *GenericWire_Constructor(AWEInstance *pAWE, INT32 *retVal, float sampleRate, UINT32 info1, UINT32 info2, UINT32 info3, const ClassDescriptor *pClass);

/**
 * @brief Wire instance constructor.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [out] retVal				to receive the error code
 * @param [in] srate				sample rate
 * @param [in] info1				packed wire info1
 * @param [in] info2				packed wire info2
 * @param [in] info3				packed wire info3
 * @return							constructed object or NULL on error
 */
/*private*/ InstanceDescriptor *ClassWire_Constructor(AWEInstance *pAWE, INT32 *retVal, INT32 srate, UINT32 info1, UINT32 info2, UINT32 info3);

/**
 * @brief Bind a wire to an I/O object.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] wireID				ID of the wire to bind
 * @param [in] pinID				pin ID of the pin to bind
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwBindIOToWire(AWEInstance *pAWE, UINT32 wireID, UINT32 pinID);

/**
 * @brief Lookup a module class by ID.
 * @param [in] classID				class to look for
 * @return							class or NULL on error
 */
/*private*/ const ModClassModule *LookupModuleClass(const AWEInstance *pAWE, UINT32 classID);

/**
 * @brief Returns the number of modules in the table.
 * @param [in] pAWE					AWE instance pointer (this)
 * @return							number of classes in module class table
 */
/*private*/ UINT32 awe_fwGetCIModuleCount(const AWEInstance *pAWE);

/**
 * @brief Get the information for the specified module.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] index				index of item to get
 * @param [out] pDescr				pointer to receive class pointer
 * @param [out] classID				pointer to receive class ID
 * @param [out] packedParameters	pointer to receive packed parameters
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwGetCIModuleInfo(const AWEInstance *pAWE, UINT32 index, UINT32 *classID, UINT32 *packedParameters);

/**
 * @brief This	executes a module's process function with profiling.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] pModule				the module to pump
 */
/*private*/ void ClassModule_Execute(AWEInstance *pAWE, ModuleInstanceDescriptor *pModule);

/**
 * @brief This is the inline "non-profiling" NP version	 of ClassModule_Execute.
 * @param [in] pModule				the module to pump
 */
/*private*/ static VEC_INLINE void ClassModule_ExecuteNP(ModuleInstanceDescriptor *pModule)
{
	pModule->pProcessFunc(pModule);
}

/**
 * @brief Base class module instance constructor.
 * @param [in] pClass				class of object to construct
 * @param [out] retVal				return error code
 * @param [in] nIO					packed wire info
 * @param [in] pWires				array of wires
 * @param [in] argCount				number of arguments
 * @param [in] args					the arguments
 * @return							constructed object or NULL on error
 *
 * This function must be called first to construct the base module, then derived
 * classes may operate on the constructed object further.
 */
/*private*/ ModInstanceDescriptor *BaseClassModule_Constructor(
		const ModClassModule * FW_RESTRICT pClass,
		INT32 * FW_RESTRICT retVal,
		UINT32 nIO,
		WireInstance * const * FW_RESTRICT pWires,
		size_t argCount,
		const Sample * FW_RESTRICT args);

/**
 * @brief Base class module instance constructor.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] pClass				class of object to construct
 * @param [out] retVal				return error code
 * @param [in] nIO					packed wire info
 * @param [in] pWireIDs				array of wire IDs
 * @param [in] argCount				number of arguments
 * @param [in] args					the arguments
 * @return							constructed object or NULL on error
 *
 * This function must be called first to construct the base module, then derived
 * classes may operate on the constructed object further.
 */
/*private*/ ModInstanceDescriptor *BaseClassModule_Constructor2(AWEInstance *pAWE,
		const ModClassModule * FW_RESTRICT pClass,
		INT32 * FW_RESTRICT retVal,
		UINT32 nIO,
		WireInstance * const * FW_RESTRICT pWires,
		size_t argCount,
		const Sample * FW_RESTRICT args);

/**
 * @brief Generic module instance constructor.
 * @param [in] classID				class ID of object to construct
 * @param [out] retVal				return error code
 * @param [in] nIO					packed wire info
 * @param [in] pWires				array of wires
 * @param [in] argCount				number of arguments
 * @param [in] args					the arguments
 * @return							constructed object or NULL on error
 */
/*private*/ ModInstanceDescriptor *ClassModule_Constructor(
		UINT32 classID,
		INT32 * FW_RESTRICT retVal,
		UINT32 nIO,
		WireInstance **pWires,
		size_t argCount,
		const Sample * FW_RESTRICT args);

/**
 * @brief Generic module instance constructor.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] classID				class ID of object to construct
 * @param [out] retVal				return error code
 * @param [in] nIO					packed wire info
 * @param [in] pWires				array of wires
 * @param [in] argCount				number of arguments
 * @param [in] args					the arguments
 * @return							constructed object or NULL on error
 */
/*private*/ ModInstanceDescriptor *ClassModule_Constructor3(
		AWEInstance *pAWE,
		UINT32 classID,
		INT32 * FW_RESTRICT retVal,
		UINT32 nIO,
		WireInstance **pWires,
		size_t argCount,
		const Sample * FW_RESTRICT args);

/**
 * @brief Generic module instance constructor.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] classID				class ID of object to construct
 * @param [out] retVal				return error code
 * @param [in] nIO					packed wire info
 * @param [in] pWireIDs				array of wire IDs
 * @param [in] argCount				number of arguments
 * @param [in] args					the arguments
 * @return							constructed object or NULL on error
 */
/*private*/ ModInstanceDescriptor *ClassModule_Constructor2(
		AWEInstance *pAWE,
		UINT32 classID,
		INT32 * FW_RESTRICT retVal,
		UINT32 nIO,
		const UINT32 *pWireIDs,
		size_t argCount,
		const Sample * FW_RESTRICT args);

/**
 * @brief Set the given module to active=0, bypass=1, mute=2, inactive=other.
 * @param [in] pModule				nmodule to set state to
 * @param [in] state				state to set
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwSetModuleState(ModuleInstanceDescriptor *pModule, INT32 state);

/**
 * @brief Get the given modules state: active=0, bypass=1, mute=2, inactive=3.
 * @param [in] pModule				module to get state from
 * @return							state, or -ve error code
 */
/*private*/ INT32 awe_fwGetModuleState(const ModuleInstanceDescriptor *pModule);

/**
 * @brief Pump the specified module.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] pModule				module to pump
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwPumpModule(AWEInstance *pAWE, ModuleInstanceDescriptor *pModule);

/**
 * @brief Pump audio
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] layoutNo				layout to pump
 * @return							0 for success, negative for error, 1 for deferred process needed
 */
INT32 awe_audioPump(AWEInstance *pAWE, UINT32 layoutNo);

/**
 * @brief Deliver an event to the specified module.
 * @param [in] pModule				module to deliver to
 * @param [in] eventCode			event to deliver
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwModuleEvent(ModuleInstanceDescriptor *pModule, INT32 eventCode);

/**
 * @brief This must be called before doing anything else with AWE.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] clock_domain			when non-zero, the clock domain of this core
 *
 * Sets all the I/O pins to a known state.
 */
void awe_fwInitIoPins(void *pAWE, UINT32 clock_domain);

/**
 * @brief Initialize the Layout IO wire buffers once layout is created
 * @param [in] pAWE					AWE instance pointer (this)
 */
void awe_fwAudioInitIO(const void *pAWE);

/**
 * @brief Layout instance constructor.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [out] retVal				error code
 * @param [in] nModules				number of modules to allocate space for
 * @param [in] nDivider				layout divider to use
 * @return							constructed object or NULL on error
 */
/*private*/ InstanceDescriptor *ClassLayout_Constructor(AWEInstance *pAWE, INT32 *retVal, INT32 nModules, INT32 nDivider);

/**
 * @brief Add one or more modules to a layout.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] argCount				argument count
 * @param [in] args					the arguments
 * @return							0 or error code
 *
 * The arguments are:
 *
 *	args[0] - ID of the layout
 *	args[1] - offset in module table
 *	args[2+] - array of module IDs
 *
 */
/*private*/ INT32 awe_fwAddModuleToLayout(const AWEInstance *pAWE, UINT32 argCount, const Sample *args);

/**
 * @brief Generic layout pump function.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] pInstance			layout to pump
 * @param [out] pInstCycles			pointer to receive profile cycles
 * @return							0 or error code
 */
/*private*/ void ClassLayout_Process(AWEInstance *pAWE, LayoutInstance *pLayout, UINT32 *pInstCycles, UINT32 *pTimePerProcess);

/**
 * @brief Master framework pump function.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] layout_no			the layout index to pump
 * @return							0 or error code
 */
INT32 awe_fwPump(void *pAWE, UINT32 layout_no);

/**
 * Master framework tick function.
 * @param [in] pAWE					AWE instance pointer (this)
 * @return							0 or error code
 */
INT32 awe_fwTick(void *pAWE);

/**
 * Call a module's set function if it has one.
 * @param [in] pModule				module pointer
 * @param [in] mask					mask to pass to set call
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwSetCall(ModuleInstanceDescriptor *pModule, UINT32 mask);

/*private*/ INT32 awe_fwSetCall2(const AWEInstance *pAWE, UINT32 moduleID, UINT32 mask);

/**
 * Call a module's get function if it has one.
 * @param [in] pModule				module pointer
 * @param [in] mask					mask to pass to set call
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwGetCall(ModuleInstanceDescriptor *pModule, UINT32 mask);

/*private*/ INT32 awe_fwGetCall2(const AWEInstance *pAWE, UINT32 moduleID, UINT32 mask);

/**
 * @brief Return the profiling information for a specified layout.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] layoutNumber			the layout index to query
 * @param [out] pAverageCycles		pointer to receive average cycles * 128
 * @param [out] pTimePerProcess		pointer to receive total cycles * 128
 * @return							0 or error code
*/
/*private*/ INT32 awe_fwGetProfileValues(const AWEInstance *pAWE, INT32 layoutNumber, UINT32 *pAverageCycles, UINT32 *pTimePerProcess);

/**
 * @brief Clear the profiling information for a specified layout.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] layoutNumber			the layout index to clear
 * @return							0 or error code
 */
/*private*/ INT32 awe_fwClearProfileValues(const AWEInstance *pAWE, INT32 layoutNumber);

/**
 * @brief Create a table for O(1) object lookup.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] maxID				max ID to size table for
 * @return							zero or error code
 */
/*private*/ INT32 awe_fwcreateLookupTable(AWEInstance *pAWE, UINT32 maxID);

/**
 * @brief Set an object pointer in the lookup table.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] ID					ID to set
 * @param [in] pObject				object pointer to set
 * @return							zero or error code
 */
/*private*/ void awe_fwSetIDPointer(const AWEInstance *pAWE, UINT32 id, InstanceDescriptor *pObject);

/**
 * @brief Lookup an object by ID.
 * @param [in] pAWE					AWE instance pointer (this)
 * @param [in] id					ID to lookup
 * @return							object pointer or NULL if not found
 */
/*private*/ void *awe_fwLookupID(const AWEInstance *pAWE, UINT32 id);

/**
 * @brief Update the object lookup table.
 * @param [in] pAWE					AWE instance pointer (this)
 *
 * Recreates the associations between ID and pointer after changing object IDs.
 */
/*private*/ INT32 awe_fwUpdateLookupTable(AWEInstance *pAWE);

/**
 * @brief Get the head of the I/O list.
 * @param [in] pAWE					AWE instance pointer (this)
 * @return							head of I/O list
 */
/*private*/ InstanceDescriptor *awe_fwGetIOPinsHead(AWEInstance *pAWE);

/**
 * @brief Get the head of the layout list.
 * @param [in] pAWE					AWE instance pointer (this)
 * @return							head of layout list
 */
/*private*/ LayoutInstance *awe_fwGetLayoutHead(AWEInstance *pAWE);

/**
 * @brief Convert a file to one in the path as needed
 * @param [out] outFile				output buffer
 * @param [in] outFileSize			size of output buffer
 * @param [in] file					file to check
 * @return							1 if exists with outFile containing path, 2 if directory with outFile containing path, else 0 if not found
 */
INT32 awe_fwFindFileInPath(char *outFile, UINT32 outFileSize, const char *file);

/**
 * @brief Set a scalar or array value by handle
 * @param pvAWE						instance pointer
 * @param handle					packed object handle
 * @param value						value(s) to set
 * @param arrayOffset				array index if array
 * @param length					number of elements if array
 * @return							error code
 */
INT32 awe_ctrlSetValue(const void *pvAWE, UINT32 handle, const void *value, INT32 arrayOffset, UINT32 length);

/**
 * @brief Get a scalar or array value by handle
 * @param pvAWE						instance pointer
 * @param handle					packed object handle
 * @param value						value(s) got
 * @param arrayOffset				array index if array
 * @param length					number of elements if array
 * @return							error code
 */
INT32 awe_ctrlGetValue(const void *pvAWE, UINT32 handle, void *value, INT32 arrayOffset, UINT32 length);

/**
 * @brief Set the status of a module.
 * @param pvAWE						instance pointer
 * @param handle					packed object handle
 * @param status					status to set
 * @return							error code
 */
INT32 awe_ctrlSetStatus(const void *pvAWE, UINT32 handle, UINT32 status);

/**
 * @brief Get the status of a module.
 * @param pvAWE						instance pointer
 * @param handle					packed object handle
 * @param status					status to get
 * @return							error code
 */
INT32 awe_ctrlGetStatus(const void *pvAWE, UINT32 handle, UINT32 *status);

/**
 * @brief Set a scalar or array value by handle with mask
 * @param pvAWE						instance pointer
 * @param handle					packed object handle
 * @param value						value(s) to set
 * @param arrayOffset				array index if array
 * @param length					number of elements if array
 * @param mask						mask to use - do not call set function
 * @return							error code
 */
INT32 awe_ctrlSetValueMask(const void *pvAWE, UINT32 handle, const void *value, INT32 arrayOffset, UINT32 length, UINT32 mask);

/**
 * @brief Get a scalar or array value by handle with mask
 * @param pvAWE						instance pointer
 * @param handle					packed object handle
 * @param value						value(s) got
 * @param arrayOffset				array index if array
 * @param length					number of elements if array
 * @param mask						mask to use - 0 do not call get function
 * @return							error code
 */
INT32 awe_ctrlGetValueMask(const void *pvAWE, UINT32 handle, void *value, INT32 arrayOffset, UINT32 length, UINT32 mask);

/**
 * @brief Initialize packet communications.
 * @param[in] packet_buf			set pointer to packet buffer
 * @param[in] reply_buf				set pointer to reply buffer
 * @param[in] buf_length			set packet buffer length in words
 */
//void awe_fwPacketInit(void *pvAWE, UINT32 *packet_buf, UINT32 *reply_buf, UINT32 buf_length);

/**
 * @brief Initialize packet communications.
 * @param[out] packet_buf			return pointer to packet buffer
 * @param [out] reply_buf			return pointer to reply buffer
 * @param[out] buf_length			return packet buffer length in words
 */
void awe_fwGetPacketInfo(const void *pvAWE, UINT32 **packet_buf, UINT32 **reply_buf, UINT32 *buf_length);

/**
 * @brief Check if this instance is running.
 * @param pvAWE						instance pointer
 * @return							non-zero if it is
 */
INT32 awe_audioIsStarted(const void *pvAWE);

/** Type of data for I/O. */
typedef enum _SampleType
{
	/** Data is 16 bit PCM. */
	Sample16bit,

	/** Data is 24 bit PCM aligned to the low bit in 32 bit buffers. */
	Sample24bit_low,

	/** Data is 24 bit PCM aligned to the high bit in 32 bit buffers. */
	Sample24bit_high,

	/** Data is 32 bit PCM . */
	Sample32bit
} SampleType;

/**
 * @brief Import samples from a user buffer to a channel
 * @param pvAWE						AWE instance
 * @param inSamples					samples to read from
 * @param inStride					input buffer stride
 * @param channel					channel to write to
 * @param inType					type of input data
 * @return							0 on success
 */
INT32 awe_audioImportSamples(const void *pvAWE, const void *inSamples, INT32 inStride, INT32 channel, SampleType inType);

/**
 * @brief Export samples to a user buffer from a channel
 * @param pvAWE						AWE instance
 * @param outSamples				samples to write to
 * @param outStride					output buffer stride
 * @param channel					channel to read from
 * @param outType					type of output data
 * @return							0 on success
 */
INT32 awe_audioExportSamples(const void *pvAWE,  void *outSamples, INT32 outStride, INT32 channel, SampleType outType);

/**
 * @brief Test if AWE is ready to run
 * @param pvAWE						AWE instance
 * @return							bit vector of threads to run - 0 if none
 */
INT32 awe_audioGetPumpMask(const void *pvAWE);

/** Class descriptor for layouts. */
extern const ClassDescriptor LayoutClass_Descriptor;

/** Class descriptor for wires. */
extern const ClassDescriptor WireClass_Descriptor;

/**
 * @brief Handle a packet for any core
 * @return							error code
 */
//INT32 awe_pltPacketProcess(AWEInstance * pAWE);

INT32 awe_packetProcessUnified(AWEInstance * pAWE);

/**
 * @brief Initialize the instance.
 * @param pvAWE						instance to initialize
 * @param initInfo					how to initialize
 */
INT32 awe_fwInit(void *pvAWE);

/** Used only by SMP. */
INT32 awe_pltInitCore(const void *pvAWE);

/** Used only by SMP. */
INT32 awe_pltDoSubLayout(const void *pvAWE);

/** Initialize the system. */
INT32 awe_init(AWEInstance *pAWE);

///** Initialize the system for packet routing only */
//INT32 awe_initNoAudio(AWEInstance *pAWE);

/** Tick the system */
//INT32 awe_tick(AWEInstance *pAWE);

/** Process an Audio Weaver command */
INT32 awe_processCmd(AWEInstance * pAWE);

/** Process a newly received tuning packet */
INT32 awe_packetProcess(AWEInstance * pAWE);

///** Do deferred processing OLLDDD */
//void awe_deferredProcess(AWEInstance * pAWE);

/** Do deferred processing */
INT32 awe_deferredSetCall(AWEInstance * pAWE);

/**Check if the layout is valid*/
INT32 awe_layoutIsValid(const void *pvAWE);

/**
 * @brief Initialize the file system.
 * @param pAWE						instance to initialize
 * @param pAWEFlashFSInstance       the file system instance
 */
void awe_initFlashFS(AWEInstance * pAWE, AWEFlashFSInstance * pAWEFlashFSInstance);

#ifndef NULL
#define NULL 0
#endif

#ifndef SUCCESS
#define SUCCESS 1
#endif

#ifndef FAILURE
#define FAILURE 0
#endif

#ifdef __GNUC__
  #if __GNUC__ >= 4
	#define DLL_PUBLIC __attribute__ ((visibility("default")))
  #else
	#define DLL_PUBLIC
  #endif
#else
  #define DLL_PUBLIC
#endif

#ifdef	__cplusplus
}
#endif

#if defined(WIN32)
#define FIXNAME(x)		x
#else
#define PASTE(x, y)		x ## y
#define EVAL(x, y)		PASTE(x, y)
#define FIXNAME(x)		EVAL(MODULE_PREFIX, x)
#endif

#define AWE_UNUSED_VARIABLE(x)	((void)(x))

#endif	/* _FRAMEWORK_H */

