function M = tutorial_peaking_filter_module(NAME)
% M = tutorial_peaking_filter_module(NAME)
% Example showing how to implement a peaking filter in Audio Weaver.
% Arguments:
%    NAME - name of the module.

% Copyright 2020.  DSP Concepts, Inc.  All Rights Reserved.

% ----------------------------------------------------------------------
% Create the module object
% ----------------------------------------------------------------------

M = awe_module('TutorialPeakingFilter', 'Multichannel Biquad filter');
if (nargin == 0)
    return;
end
M.name = NAME;

% ----------------------------------------------------------------------
% Assign module functions.
% Process function is optional and used for regression tests.
% ----------------------------------------------------------------------

M.processFunc = @tutorial_peaking_filter_process;
M.testHarnessFunc = @test_tutorial_peaking_filter;
M.preBuildFunc = @tutorial_peaking_filter_prebuild;
M.setFunc = @tutorial_peaking_filter_set;
M.freqRespFunc = @tutorial_peaking_filter_freq_response;

% ----------------------------------------------------------------------
% Add input and output pins
% ----------------------------------------------------------------------

PT = new_pin_type([], [], []);
add_pin(M, 'input', 'in', 'Input signal', PT);
add_pin(M, 'output', 'out', 'Output signal', PT);

% ----------------------------------------------------------------------
% Add module variables
% ----------------------------------------------------------------------

add_variable(M, 'freq', 'float', 250, 'parameter', 'Cutoff frequency of the filter, in Hz');
M.freq.units = 'Hz';
% Set default values here.  Actual M.freq.range is set in the update function.
M.freq.range=[10 20000 0.1];

add_variable(M, 'gain', 'float', 0, 'parameter', 'Amount of boost or cut to apply');
M.gain.units = 'dB';
M.gain.range=[-24 24 0.1];

add_variable(M, 'Q', 'float', 1, 'parameter', 'Specifies the Q of the filter');
M.Q.range=[0 20 0.1];

add_variable(M, 'smoothingTime', 'float', 10, 'parameter', 'Time constant of the smoothing process');
M.smoothingTime.units = 'msec';
M.smoothingTime.range = [0 1000];

add_variable(M, 'smoothingCoeff', 'float', 1, 'derived', 'Smoothing coefficient. This is computed based on the smoothingTime, sample rate, and block size of the module');
M.smoothingCoeff.isHidden = 1;

add_array(M, 'coeffs', 'float', [1; 0; 0; 0; 0], 'derived', 'Filter coefficients [b0; b1; b2; a1; a2]');
M.coeffs.isHidden = 1;

add_array(M, 'currentCoeffs', 'float', [1; 0; 0; 0; 0], 'state', 'Smoothed filter coefficients [b0; b1; b2; a1; a2]');
M.currentCoeffs.isHidden = 1;

% Initialize the state array for a single channel.  This will be updated
% in the module's prebuild function below
add_array(M, 'state', 'float', [0;0], 'state', 'State variables. 2 per channel.');
M.state.arrayHeap = 'AWE_HEAP_FAST2SLOW';
M.state.arraySizeConstructor = 'ClassWire_GetChannelCount(pWires[0]) * 2 * sizeof(FLOAT32)';
M.state.isHidden = 1;

% Call the set function.  This provides a reasonable starting value for the
% smoothing Coeff
M = update(M);

% ----------------------------------------------------------------------
% Code generation details
% ----------------------------------------------------------------------

awe_addcodemarker(M, 'processFunction', 'Insert:InnerTutorialPeakingFilter_Process.c');
awe_addcodemarker(M, 'setFunction', 'Insert:\InnerTutorialPeakingFilter_Set.c');

% This means that the processing can be done in-place
M.wireAllocation = 'across';

% ----------------------------------------------------------------------
% Module documentation
% ----------------------------------------------------------------------

awe_addcodemarker(M, 'discussion', {'Peaking filter with configurable frequency, gain, and Q parameters. ', ...
'The module operates on multichannel data and is smoothly updating. ', ...
'Internally, the module uses RBF design equations.', ...
'', ...
'This module is an example to be used with the documentation.'});

% ----------------------------------------------------------------------
% Add the inspector information
% ----------------------------------------------------------------------

M.freq.guiInfo.attribStr='mapping=log useticks=1 fixedticks=11';
add_control(M, '.freq');

add_control(M, '.Q', 'below');

M.gain.guiInfo.controlType='slider';
add_control(M, '.gain', 'topright');

% ----------------------------------------------------------------------
% Module browser information
% ----------------------------------------------------------------------

M.moduleBrowser.path = 'Tutorial';
M.moduleBrowser.image = '../images/Tutorial.bmp';
M.moduleBrowser.searchTags = 'peaking';
M.shapeInfo.basicShape = 'rectangle';
M.shapeInfo.legend = 'Biquad';

% ----------------------------------------------------------------------
% Pin function.  This changes the size of the state variables based
% on the number of channels in the input.  We also make the output
% pin the same type as the input pin.
% ----------------------------------------------------------------------

function M = tutorial_peaking_filter_prebuild(M)

% Have the module start in a converged state
M.currentCoeffs = M.coeffs;

% Set the size of the state variables based on the number of channels.
M.state.size = [2 M.inputPin{1}.type.numChannels];
M.state = zeros(2, M.inputPin{1}.type.numChannels);

% Propogate the input pin type to the output pin
% This copies numChannels, blockSize, and sampleRate
M.outputPin{1}.type = M.inputPin{1}.type;

return;

% ----------------------------------------------------------------------
% Set function.
% ----------------------------------------------------------------------

function M = tutorial_peaking_filter_set(M)

% Get the sampleRate and blockSize from the input pin
sampleRate = M.inputPin{1}.type.sampleRate;
blockSize = M.inputPin{1}.type.blockSize;

% Taken from Audio-EQ-Cookbook.txt, by Robert Bristow-Johnson
w0 = 2 * pi * M.freq / sampleRate;
A = undb20(M.gain * 0.5);

if (M.Q >= 0)
    alpha = sin(w0) / (2 * M.Q);
else
    alpha = sin(w0) * sinh((log(2)/2)*(-M.Q)*w0/sin(w0));
end

b0 =   1 + alpha*A;
b1 =  -2*cos(w0);
b2 =   1 - alpha*A;
a0 =   1 + alpha/A;
a1 =  -2*cos(w0);
a2 =   1 - alpha/A;
 
% Normalize so that the a0 coefficient equals 1.0
C = [b0; b1; b2; a1; a2] / a0;

M.coeffs = C;

% Compute the smoothing coefficient
M.smoothingCoeff = 1.0 - exp(-1.0/((sampleRate / blockSize) * 0.001 * M.smoothingTime));

return;

% ----------------------------------------------------------------------
% Computes the frequency response
% ----------------------------------------------------------------------

function H_OUT = tutorial_peaking_filter_freq_response(M, H_IN, W)

% Only a single input pin
H_IN = H_IN{1};

b = M.coeffs(1:3);
a = [1; M.coeffs(4:5)];
H = freqz(b, a, W);

numChannels = size(H_IN, 2);
H = repmat(H, 1, numChannels);
H_OUT = H_IN .* H;

% Make it a cell array.
H_OUT={H_OUT};

return;