function M = tutorial_biquad_mono_module(NAME)
% M = tutorial_biquad_mono_module(NAME)
% Example showing how to implement a mono Biquad filter in Audio Weaver.
% Arguments:
%    NAME - name of the module.

% Copyright 2020.  DSP Concepts, Inc.  All Rights Reserved.

% ----------------------------------------------------------------------
% Create the module object
% ----------------------------------------------------------------------

M = awe_module('TutorialBiquadMono', 'Mono Biquad filter');
if (nargin == 0)
    return;
end
M.name = NAME;

% ----------------------------------------------------------------------
% Assign module functions.
% Process function is optional and used for regression tests.
% ----------------------------------------------------------------------

M.processFunc = @tutorial_biquad_mono_process;
M.testHarnessFunc = @test_tutorial_biquad_mono;

% ----------------------------------------------------------------------
% Add input and output pins
% ----------------------------------------------------------------------

PT = new_pin_type(1, [], []);
add_pin(M, 'input', 'in', 'Input signal', PT);
add_pin(M, 'output', 'out', 'Output signal', PT);

% ----------------------------------------------------------------------
% Add module variables
% ----------------------------------------------------------------------

add_variable(M, 'b0', 'float', 1, 'parameter', 'First numerator coefficient');
add_variable(M, 'b1', 'float', 0, 'parameter', 'Second numerator coefficient');
add_variable(M, 'b2', 'float', 0, 'parameter', 'Third numerator coefficient');
add_variable(M, 'a1', 'float', 0, 'parameter', 'Second denominator coefficient');
add_variable(M, 'a2', 'float', 0, 'parameter', 'Third denominator coefficient');

add_variable(M, 'wNm1', 'float', 0, 'state', 'First state variable');
add_variable(M, 'wNm2', 'float', 0, 'state', 'Second state variable');

% ----------------------------------------------------------------------
% Code generation details
% ----------------------------------------------------------------------

awe_addcodemarker(M, 'processFunction', 'Insert:InnerTutorialBiquadMono_Process.c');

% This means that the processing can be done in-place
M.wireAllocation = 'across';

% ----------------------------------------------------------------------
% Module documentation
% ----------------------------------------------------------------------

awe_addcodemarker(M, 'discussion', {'Standard 5 coefficient Biquad filter that operates on mono data. ', ...
'The module uses a Transposed Direct Form 2 implementation using the difference equation: ', ...
'', ...
'  wN = a1*wNm1 + a2*wNm2 + x[n]', ...
'y[n] = b0*wN + b1*wNm1 + b2*wNm2', ...
'wNm2 = wNm1', ...
'wNm1 = wN', ...
'', ...
'This module is an example to be used with the documentation.'});

% ----------------------------------------------------------------------
% Add the inspector information
% ----------------------------------------------------------------------

add_control(M, 'b0');
add_control(M, 'b1');
add_control(M, 'b2');
add_control(M, 'a1');
add_control(M, 'a2');

% ----------------------------------------------------------------------
% Module browser information
% ----------------------------------------------------------------------

M.moduleBrowser.path = 'Tutorial';
M.moduleBrowser.image = '../images/Tutorial.bmp';
M.moduleBrowser.searchTags = 'filter';
M.shapeInfo.basicShape = 'rectangle';
M.shapeInfo.legend = 'Biquad';



